<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Services\FinancialReportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class FinancialReportController extends Controller
{
    /**
     * خدمة التقارير المالية
     *
     * @var FinancialReportService
     */
    protected $reportService;

    /**
     * إنشاء مثيل جديد من وحدة التحكم
     *
     * @param FinancialReportService $reportService
     */
    public function __construct(FinancialReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    /**
     * تقرير الإيرادات
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function revenueReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = Auth::user()->company_id;
        $branchId = $request->branch_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $report = $this->reportService->generateRevenueReport($companyId, $startDate, $endDate, $branchId);

        return response()->json([
            'status' => 'success',
            'data' => $report,
        ]);
    }

    /**
     * تقرير المدفوعات
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function paymentsReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = Auth::user()->company_id;
        $branchId = $request->branch_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $report = $this->reportService->generatePaymentsReport($companyId, $startDate, $endDate, $branchId);

        return response()->json([
            'status' => 'success',
            'data' => $report,
        ]);
    }

    /**
     * تقرير العقود
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function contractsReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = Auth::user()->company_id;
        $branchId = $request->branch_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $report = $this->reportService->generateContractsReport($companyId, $startDate, $endDate, $branchId);

        return response()->json([
            'status' => 'success',
            'data' => $report,
        ]);
    }

    /**
     * تقرير الذمم المدينة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function accountsReceivableReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = Auth::user()->company_id;
        $branchId = $request->branch_id;

        $report = $this->reportService->generateAccountsReceivableReport($companyId, $branchId);

        return response()->json([
            'status' => 'success',
            'data' => $report,
        ]);
    }

    /**
     * تقرير كشف حساب العميل
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function customerStatementReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = Auth::user()->company_id;
        $customerId = $request->customer_id;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $report = $this->reportService->generateCustomerStatementReport($companyId, $customerId, $startDate, $endDate);

        return response()->json([
            'status' => 'success',
            'data' => $report,
        ]);
    }
}
