<?php

namespace App\Helpers;

use App\Models\InvoiceInstallment;
use App\Models\Invoice;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class InstallmentHelper
{
    /**
     * إنشاء أقساط تلقائية للفاتورة.
     *
     * @param Invoice $invoice
     * @param int $numberOfInstallments
     * @param string $startDate
     * @param string $interval
     * @param string $notes
     * @return array
     */
    public static function createInstallmentsForInvoice(Invoice $invoice, int $numberOfInstallments, string $startDate, string $interval = 'month', string $notes = '')
    {
        // التحقق من صحة المدخلات
        if ($numberOfInstallments <= 0) {
            return [
                'success' => false,
                'message' => 'عدد الأقساط يجب أن يكون أكبر من صفر.',
            ];
        }

        // حساب قيمة كل قسط
        $installmentAmount = round($invoice->total_amount / $numberOfInstallments, 2);
        $remainingAmount = $invoice->total_amount - ($installmentAmount * ($numberOfInstallments - 1));

        // تاريخ بداية الأقساط
        $dueDate = Carbon::parse($startDate);

        // بدء المعاملة
        DB::beginTransaction();

        try {
            $createdInstallments = [];

            // إنشاء الأقساط
            for ($i = 0; $i < $numberOfInstallments; $i++) {
                // القسط الأخير يأخذ المبلغ المتبقي لتجنب فروقات التقريب
                $amount = ($i == $numberOfInstallments - 1) ? $remainingAmount : $installmentAmount;

                $installment = InvoiceInstallment::create([
                    'invoice_id' => $invoice->id,
                    'amount' => $amount,
                    'paid_amount' => 0,
                    'due_date' => $dueDate->format('Y-m-d'),
                    'status' => 'pending',
                    'notes' => $notes . ' (قسط ' . ($i + 1) . '/' . $numberOfInstallments . ')',
                ]);

                $createdInstallments[] = $installment;

                // تحديث تاريخ الاستحقاق للقسط التالي
                if ($interval == 'month') {
                    $dueDate->addMonth();
                } elseif ($interval == 'week') {
                    $dueDate->addWeek();
                } elseif ($interval == 'day') {
                    $dueDate->addDay();
                } else {
                    $dueDate->addMonth();
                }
            }

            // تحديث حالة الفاتورة
            $invoice->update([
                'payment_type' => 'installments',
                'status' => 'partially_paid',
            ]);

            DB::commit();

            return [
                'success' => true,
                'message' => 'تم إنشاء ' . $numberOfInstallments . ' قسط بنجاح.',
                'installments' => $createdInstallments,
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء الأقساط: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * الحصول على إحصائيات الأقساط للشركة.
     *
     * @param int $companyId
     * @return array
     */
    public static function getInstallmentStats(int $companyId)
    {
        // إحصائيات الأقساط
        $stats = [
            'total' => 0,
            'paid' => 0,
            'pending' => 0,
            'overdue' => 0,
            'total_amount' => 0,
            'paid_amount' => 0,
            'due_amount' => 0,
            'overdue_amount' => 0,
        ];

        // الحصول على جميع الأقساط للشركة
        $allInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })->get();

        $stats['total'] = $allInstallments->count();
        $stats['total_amount'] = $allInstallments->sum('amount');
        $stats['paid_amount'] = $allInstallments->sum('paid_amount');
        $stats['due_amount'] = $stats['total_amount'] - $stats['paid_amount'];

        // تصنيف الأقساط حسب الحالة
        $stats['paid'] = $allInstallments->filter(function ($installment) {
            return $installment->isPaid();
        })->count();

        $stats['pending'] = $allInstallments->filter(function ($installment) {
            return !$installment->isPaid() && !$installment->isOverdue();
        })->count();

        $overdueInstallments = $allInstallments->filter(function ($installment) {
            return $installment->isOverdue();
        });

        $stats['overdue'] = $overdueInstallments->count();
        $stats['overdue_amount'] = $overdueInstallments->sum(function ($installment) {
            return $installment->amount - $installment->paid_amount;
        });

        return $stats;
    }

    /**
     * الحصول على الأقساط المستحقة قريباً.
     *
     * @param int $companyId
     * @param int $days
     * @param int $limit
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getUpcomingInstallments(int $companyId, int $days = 7, int $limit = 10)
    {
        return InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('status', 'pending')
            ->where('due_date', '<=', Carbon::now()->addDays($days))
            ->where('due_date', '>=', Carbon::now())
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->limit($limit)
            ->get();
    }

    /**
     * الحصول على الأقساط المتأخرة.
     *
     * @param int $companyId
     * @param int $limit
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getOverdueInstallments(int $companyId, int $limit = 10)
    {
        return InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where(function ($query) {
                $query->where('status', 'overdue')
                    ->orWhere(function ($q) {
                        $q->where('status', 'pending')
                            ->where('due_date', '<', Carbon::now());
                    });
            })
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->limit($limit)
            ->get();
    }
}
