<?php

namespace App\Exports;

use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class TechnicianPerformanceExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithStyles
{
    /**
     * بيانات التقرير.
     *
     * @var array
     */
    protected $report;

    /**
     * تاريخ بداية التقرير.
     *
     * @var \Carbon\Carbon
     */
    protected $startDate;

    /**
     * تاريخ نهاية التقرير.
     *
     * @var \Carbon\Carbon
     */
    protected $endDate;

    /**
     * إنشاء مثيل جديد للتصدير.
     *
     * @param array $report
     * @param \Carbon\Carbon $startDate
     * @param \Carbon\Carbon $endDate
     * @return void
     */
    public function __construct($report, $startDate, $endDate)
    {
        $this->report = $report;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    /**
     * الحصول على مجموعة البيانات للتصدير.
     *
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return collect($this->report['technician_performance']);
    }

    /**
     * تعيين عناوين الأعمدة.
     *
     * @return array
     */
    public function headings(): array
    {
        return [
            'اسم الفني',
            'عمليات الصيانة المكتملة',
            'عمليات الصيانة الفاشلة',
            'متوسط وقت الإنجاز (ساعة)',
            'عدد قطع الغيار المستبدلة',
            'نسبة النجاح (%)'
        ];
    }

    /**
     * تعيين قيم الأعمدة.
     *
     * @param mixed $row
     * @return array
     */
    public function map($row): array
    {
        return [
            $row['technician_name'],
            $row['completed_maintenance_count'],
            $row['failed_maintenance_count'],
            number_format($row['average_completion_time'], 1),
            $row['parts_replaced_count'],
            $row['success_rate']
        ];
    }

    /**
     * تعيين عنوان ورقة العمل.
     *
     * @return string
     */
    public function title(): string
    {
        return 'تقرير أداء الفنيين';
    }

    /**
     * تعيين أنماط الخلايا.
     *
     * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
     * @return void
     */
    public function styles(Worksheet $sheet)
    {
        // تنسيق العناوين
        $sheet->getStyle('A1:F1')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 14
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => [
                    'rgb' => 'DDDDDD'
                ]
            ]
        ]);

        // تنسيق الخلايا
        $sheet->getStyle('A1:F' . ($this->collection()->count() + 1))->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN
                ]
            ]
        ]);

        // إضافة معلومات التقرير
        $sheet->mergeCells('A' . ($this->collection()->count() + 3) . ':F' . ($this->collection()->count() + 3));
        $sheet->setCellValue('A' . ($this->collection()->count() + 3), 'ملخص تقرير أداء الفنيين');
        $sheet->getStyle('A' . ($this->collection()->count() + 3))->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 14
            ]
        ]);

        $sheet->setCellValue('A' . ($this->collection()->count() + 4), 'الفترة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 4), $this->startDate->format('Y-m-d') . ' إلى ' . $this->endDate->format('Y-m-d'));

        $sheet->setCellValue('A' . ($this->collection()->count() + 5), 'إجمالي الفنيين:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 5), $this->report['total_technicians']);

        $sheet->setCellValue('A' . ($this->collection()->count() + 6), 'إجمالي عمليات الصيانة المكتملة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 6), $this->report['total_completed_maintenance']);

        $sheet->setCellValue('A' . ($this->collection()->count() + 7), 'إجمالي عمليات الصيانة الفاشلة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 7), $this->report['total_failed_maintenance']);

        $sheet->setCellValue('A' . ($this->collection()->count() + 8), 'متوسط وقت الإنجاز:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 8), number_format($this->report['average_completion_time'], 1) . ' ساعة');
    }
}
