<?php

namespace App\Exports;

use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class MaintenanceFinancialExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithStyles
{
    /**
     * بيانات التقرير.
     *
     * @var array
     */
    protected $report;

    /**
     * تاريخ بداية التقرير.
     *
     * @var \Carbon\Carbon
     */
    protected $startDate;

    /**
     * تاريخ نهاية التقرير.
     *
     * @var \Carbon\Carbon
     */
    protected $endDate;

    /**
     * إنشاء مثيل جديد للتصدير.
     *
     * @param array $report
     * @param \Carbon\Carbon $startDate
     * @param \Carbon\Carbon $endDate
     * @return void
     */
    public function __construct($report, $startDate, $endDate)
    {
        $this->report = $report;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    /**
     * الحصول على مجموعة البيانات للتصدير.
     *
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return collect($this->report['invoices']);
    }

    /**
     * تعيين عناوين الأعمدة.
     *
     * @return array
     */
    public function headings(): array
    {
        return [
            'رقم الفاتورة',
            'العميل',
            'المصعد',
            'تاريخ الفاتورة',
            'تاريخ الاستحقاق',
            'المبلغ الفرعي',
            'الضريبة',
            'المبلغ الإجمالي',
            'الحالة',
            'ملاحظات'
        ];
    }

    /**
     * تعيين قيم الأعمدة.
     *
     * @param mixed $row
     * @return array
     */
    public function map($row): array
    {
        return [
            $row->invoice_number,
            $row->customer->name ?? 'غير محدد',
            $row->elevator->name ?? 'غير محدد',
            $row->invoice_date->format('Y-m-d'),
            $row->due_date->format('Y-m-d'),
            $row->subtotal,
            $row->tax_amount,
            $row->total_amount,
            $this->getStatusText($row->status),
            $row->notes
        ];
    }

    /**
     * تعيين عنوان ورقة العمل.
     *
     * @return string
     */
    public function title(): string
    {
        return 'التقرير المالي للصيانة';
    }

    /**
     * تعيين أنماط الخلايا.
     *
     * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
     * @return void
     */
    public function styles(Worksheet $sheet)
    {
        // تنسيق العناوين
        $sheet->getStyle('A1:J1')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 14
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => [
                    'rgb' => 'DDDDDD'
                ]
            ]
        ]);

        // تنسيق الخلايا
        $sheet->getStyle('A1:J' . ($this->collection()->count() + 1))->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN
                ]
            ]
        ]);

        // إضافة معلومات التقرير
        $sheet->mergeCells('A' . ($this->collection()->count() + 3) . ':J' . ($this->collection()->count() + 3));
        $sheet->setCellValue('A' . ($this->collection()->count() + 3), 'ملخص التقرير المالي للصيانة');
        $sheet->getStyle('A' . ($this->collection()->count() + 3))->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 14
            ]
        ]);

        $sheet->setCellValue('A' . ($this->collection()->count() + 4), 'الفترة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 4), $this->startDate->format('Y-m-d') . ' إلى ' . $this->endDate->format('Y-m-d'));

        $sheet->setCellValue('A' . ($this->collection()->count() + 5), 'إجمالي الإيرادات:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 5), $this->report['total_revenue'] . ' ر.س');

        $sheet->setCellValue('A' . ($this->collection()->count() + 6), 'عمليات الصيانة المكتملة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 6), $this->report['completed_maintenance_count']);

        $sheet->setCellValue('A' . ($this->collection()->count() + 7), 'عمليات الصيانة المعلقة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 7), $this->report['pending_maintenance_count']);

        $sheet->setCellValue('A' . ($this->collection()->count() + 8), 'عمليات الصيانة الفاشلة:');
        $sheet->setCellValue('B' . ($this->collection()->count() + 8), $this->report['failed_maintenance_count']);
    }

    /**
     * الحصول على نص حالة الفاتورة.
     *
     * @param string $status
     * @return string
     */
    private function getStatusText($status)
    {
        switch ($status) {
            case 'paid':
                return 'مدفوعة';
            case 'partially_paid':
                return 'مدفوعة جزئياً';
            case 'unpaid':
                return 'غير مدفوعة';
            default:
                return $status;
        }
    }
}
