<?php

namespace App\Exports;

use App\Models\CashTransaction;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class DailyMovementExport implements FromCollection, WithHeadings, WithMapping, WithTitle, ShouldAutoSize, WithStyles
{
    protected $date;
    protected $cashRegisterId;

    public function __construct($date, $cashRegisterId = null)
    {
        $this->date = $date;
        $this->cashRegisterId = $cashRegisterId;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = CashTransaction::where('company_id', Auth::user()->company_id)
            ->whereDate('created_at', $this->date);

        if ($this->cashRegisterId) {
            $query->where(function ($q) {
                $q->where('cash_register_id', $this->cashRegisterId)
                    ->orWhere('related_cash_register_id', $this->cashRegisterId);
            });
        }

        return $query->with(['cashRegister', 'relatedCashRegister', 'user'])
            ->orderBy('created_at')
            ->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            '#',
            'الوقت',
            'نوع المعاملة',
            'الخزينة',
            'الخزينة المرتبطة',
            'المبلغ',
            'الوصف',
            'المستخدم',
            'المرجع'
        ];
    }

    /**
     * @param mixed $row
     *
     * @return array
     */
    public function map($transaction): array
    {
        static $i = 0;
        $i++;

        $type = '';
        if ($transaction->type == 'deposit') {
            $type = 'إيداع';
        } elseif ($transaction->type == 'withdrawal') {
            $type = 'سحب';
        } elseif ($transaction->type == 'transfer') {
            $type = 'تحويل';
        }

        return [
            $i,
            $transaction->created_at->format('H:i:s'),
            $type,
            $transaction->cashRegister->name,
            $transaction->relatedCashRegister ? $transaction->relatedCashRegister->name : '-',
            $transaction->amount,
            $transaction->description,
            $transaction->user->name,
            $transaction->reference_number
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'تقرير الحركة اليومية - ' . $this->date;
    }

    /**
     * @param Worksheet $sheet
     */
    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
}
