<?php

namespace App\Exports;

use App\Models\ChartOfAccount;
use App\Models\JournalEntryItem;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class AccountBalanceExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize, WithEvents
{
    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // الحصول على الحسابات الفرعية فقط (التي ليس لها حسابات فرعية)
        $query = ChartOfAccount::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $companyId);
        }

        $accounts = $query->where('is_active', true)
            ->where('is_parent', false)
            ->orderBy('account_code')
            ->get();

        // الحصول على تاريخ آخر حركة لكل حساب
        foreach ($accounts as $account) {
            // الحصول على آخر حركة للحساب
            $lastMovement = JournalEntryItem::where('account_id', $account->id)
                ->join('journal_entries', 'journal_entries.id', '=', 'journal_entry_items.journal_entry_id')
                ->where('journal_entries.is_posted', true)
                ->orderBy('journal_entries.entry_date', 'desc')
                ->select('journal_entries.entry_date')
                ->first();

            $account->last_movement_date = $lastMovement ? $lastMovement->entry_date : null;

            // تحديد نوع الرصيد (مدين/دائن) بناءً على نوع الحساب
            $balance = $account->current_balance;
            $account->balance_type = '';

            if (in_array($account->account_type, ['asset', 'expense'])) {
                // الأصول والمصروفات: الرصيد الموجب يكون مدين، والسالب يكون دائن
                if ($balance > 0) {
                    $account->balance_type = 'مدين';
                } elseif ($balance < 0) {
                    $account->balance_type = 'دائن';
                    $account->display_balance = abs($balance);
                } else {
                    $account->display_balance = 0;
                }
            } else {
                // الخصوم والإيرادات وحقوق الملكية: الرصيد الموجب يكون دائن، والسالب يكون مدين
                if ($balance > 0) {
                    $account->balance_type = 'دائن';
                } elseif ($balance < 0) {
                    $account->balance_type = 'مدين';
                    $account->display_balance = abs($balance);
                } else {
                    $account->display_balance = 0;
                }
            }

            // إذا لم يتم تعيين display_balance، استخدم القيمة الأصلية
            if (!isset($account->display_balance)) {
                $account->display_balance = $balance;
            }
        }

        // تصفية الحسابات التي لها رصيد فقط
        return $accounts->filter(function ($account) {
            return $account->display_balance != 0;
        });
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'رقم الحساب',
            'اسم الحساب',
            'تاريخ آخر حركة',
            'الرصيد النهائي',
            'نوع الرصيد',
            'العملة'
        ];
    }

    /**
     * @param mixed $row
     * @return array
     */
    public function map($row): array
    {
        return [
            $row->account_code,
            $row->name,
            $row->last_movement_date ? date('Y-m-d', strtotime($row->last_movement_date)) : 'لا توجد حركات',
            number_format($row->display_balance, 2),
            $row->balance_type,
            $row->currency
        ];
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق الصف الأول (العناوين)
            1 => [
                'font' => [
                    'bold' => true,
                    'size' => 14,
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => [
                        'rgb' => 'E2EFDA',
                    ],
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                // تعيين اتجاه الورقة من اليمين إلى اليسار
                $event->sheet->getDelegate()->setRightToLeft(true);

                // تطبيق حدود على جميع الخلايا
                $lastColumn = $event->sheet->getDelegate()->getHighestColumn();
                $lastRow = $event->sheet->getDelegate()->getHighestRow();
                $range = 'A1:' . $lastColumn . $lastRow;

                $event->sheet->getDelegate()->getStyle($range)->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);

                // تجميد الصف الأول
                $event->sheet->getDelegate()->freezePane('A2');

                // إضافة صف للإجمالي في النهاية
                $totalRow = $lastRow + 1;
                $event->sheet->getDelegate()->setCellValue('A' . $totalRow, 'الإجمالي');
                $event->sheet->getDelegate()->mergeCells('A' . $totalRow . ':C' . $totalRow);

                // تنسيق صف الإجمالي
                $event->sheet->getDelegate()->getStyle('A' . $totalRow . ':' . $lastColumn . $totalRow)->getFont()->setBold(true);
                $event->sheet->getDelegate()->getStyle('A' . $totalRow . ':' . $lastColumn . $totalRow)->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->getStartColor()->setRGB('F2F2F2');
            },
        ];
    }
}
