<?php

namespace App\Console\Commands;

use App\Services\NotificationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SendPendingNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notifications:send-pending';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'إرسال جميع الإشعارات المعلقة';

    /**
     * خدمة الإشعارات.
     *
     * @var \App\Services\NotificationService
     */
    protected $notificationService;

    /**
     * Create a new command instance.
     *
     * @param  \App\Services\NotificationService  $notificationService
     * @return void
     */
    public function __construct(NotificationService $notificationService)
    {
        parent::__construct();
        $this->notificationService = $notificationService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('بدء إرسال الإشعارات المعلقة...');
        Log::info('تشغيل أمر إرسال الإشعارات المعلقة');

        try {
            // Get count of pending notifications before sending
            $pendingCount = \App\Models\Notification::where('status', 'pending')->count();
            $this->info("عدد الإشعارات المعلقة قبل الإرسال: {$pendingCount}");
            Log::info("عدد الإشعارات المعلقة قبل الإرسال: {$pendingCount}");

            // Send pending notifications
            $count = $this->notificationService->sendPendingNotifications();

            // Get count after sending
            $remainingCount = \App\Models\Notification::where('status', 'pending')->count();
            $sentCount = \App\Models\Notification::where('status', 'sent')
                ->whereDate('sent_at', today())->count();

            $this->info("تم إرسال {$count} إشعار معلق بنجاح.");
            $this->info("عدد الإشعارات المتبقية: {$remainingCount}");
            $this->info("إجمالي الإشعارات المرسلة اليوم: {$sentCount}");

            Log::info("اكتمل أمر إرسال الإشعارات المعلقة", [
                'sent' => $count,
                'remaining' => $remainingCount,
                'total_sent_today' => $sentCount
            ]);

            return 0;
        } catch (\Exception $e) {
            $errorMessage = 'فشل إرسال الإشعارات المعلقة: ' . $e->getMessage();
            $this->error($errorMessage);

            Log::error($errorMessage, [
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return 1;
        }
    }
}
