<?php

namespace App\Console\Commands;

use App\Models\InvoiceInstallment;
use App\Models\User;
use App\Notifications\InstallmentDueNotification;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class SendInstallmentReminders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'installments:send-reminders';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'إرسال تذكيرات للأقساط المستحقة قريباً والمتأخرة';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('بدء إرسال تذكيرات الأقساط...');

        // الأقساط المستحقة خلال 7 أيام
        $upcomingInstallments = InvoiceInstallment::where('status', 'pending')
            ->where('due_date', '>=', Carbon::now()->startOfDay())
            ->where('due_date', '<=', Carbon::now()->addDays(7)->endOfDay())
            ->with(['invoice', 'invoice.customer', 'invoice.company'])
            ->get();

        $this->info("تم العثور على {$upcomingInstallments->count()} قسط مستحق خلال 7 أيام.");

        foreach ($upcomingInstallments as $installment) {
            $daysRemaining = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false);

            // إرسال إشعار للعميل إذا كان لديه حساب مستخدم
            $customer = $installment->invoice->customer;
            if ($customer->user_id) {
                $user = User::find($customer->user_id);
                if ($user) {
                    $user->notify(new InstallmentDueNotification($installment, $daysRemaining));
                }
            }

            // إرسال إشعار للمستخدمين المسؤولين عن متابعة الأقساط
            $companyId = $installment->invoice->company_id;
            $admins = User::whereHas('roles', function ($query) {
                $query->whereIn('name', ['super_admin', 'admin', 'accountant']);
            })
                ->where('company_id', $companyId)
                ->get();

            Notification::send($admins, new InstallmentDueNotification($installment, $daysRemaining));

            $this->info("تم إرسال تذكير للقسط رقم {$installment->id} (متبقي {$daysRemaining} يوم).");
        }

        // الأقساط المتأخرة
        $overdueInstallments = InvoiceInstallment::where(function ($query) {
            $query->where('status', 'overdue')
                ->orWhere(function ($q) {
                    $q->where('status', 'pending')
                        ->where('due_date', '<', Carbon::now()->startOfDay());
                });
        })
            ->where('paid_amount', '<', \DB::raw('amount'))
            ->with(['invoice', 'invoice.customer', 'invoice.company'])
            ->get();

        $this->info("تم العثور على {$overdueInstallments->count()} قسط متأخر.");

        foreach ($overdueInstallments as $installment) {
            $daysOverdue = Carbon::now()->startOfDay()->diffInDays($installment->due_date, false) * -1;

            // إرسال إشعار للعميل إذا كان لديه حساب مستخدم
            $customer = $installment->invoice->customer;
            if ($customer->user_id) {
                $user = User::find($customer->user_id);
                if ($user) {
                    $user->notify(new InstallmentDueNotification($installment, $daysOverdue * -1));
                }
            }

            // إرسال إشعار للمستخدمين المسؤولين عن متابعة الأقساط
            $companyId = $installment->invoice->company_id;
            $admins = User::whereHas('roles', function ($query) {
                $query->whereIn('name', ['super_admin', 'admin', 'accountant']);
            })
                ->where('company_id', $companyId)
                ->get();

            Notification::send($admins, new InstallmentDueNotification($installment, $daysOverdue * -1));

            $this->info("تم إرسال تذكير للقسط المتأخر رقم {$installment->id} (متأخر {$daysOverdue} يوم).");
        }

        $this->info('تم الانتهاء من إرسال تذكيرات الأقساط بنجاح.');

        // تسجيل العملية في السجل
        Log::info("تم إرسال تذكيرات لـ {$upcomingInstallments->count()} قسط مستحق و {$overdueInstallments->count()} قسط متأخر.");

        return 0;
    }
}
