<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CashTransaction;
use App\Models\JournalEntry;
use App\Http\Controllers\CashTransactionController;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RecreateDepositJournalEntries extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'journal:recreate-deposits {date? : التاريخ المحدد بتنسيق Y-m-d}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'إعادة إنشاء قيود المحاسبة للإيداعات النقدية التي لم يتم إنشاء قيود محاسبية لها';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $date = $this->argument('date');

        $this->info('بدء إعادة إنشاء قيود المحاسبة للإيداعات النقدية...');

        // البحث عن الإيداعات النقدية التي لم يتم إنشاء قيود محاسبية لها
        $query = CashTransaction::where('type', 'deposit')
            ->whereNotExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('cash_transaction_entries')
                    ->whereRaw('cash_transaction_entries.cash_transaction_id = cash_transactions.id');
            });

        // تطبيق فلتر التاريخ إذا تم تحديده
        if ($date) {
            $query->whereDate('created_at', $date);
            $this->info("تم تحديد التاريخ: {$date}");
        }

        $deposits = $query->get();

        $this->info("تم العثور على {$deposits->count()} إيداع بدون قيود محاسبية.");

        if ($deposits->isEmpty()) {
            $this->info('لا توجد إيداعات تحتاج إلى إعادة إنشاء قيود محاسبية.');
            return 0;
        }

        $controller = new CashTransactionController();
        $successCount = 0;
        $errorCount = 0;

        foreach ($deposits as $deposit) {
            $this->info("معالجة الإيداع رقم {$deposit->id} بتاريخ {$deposit->created_at}...");

            try {
                DB::beginTransaction();

                // استدعاء دالة إنشاء قيد المحاسبة
                $entry = $controller->createJournalEntryForDeposit($deposit);

                if ($entry) {
                    $this->info("تم إنشاء قيد المحاسبة رقم {$entry->id} للإيداع رقم {$deposit->id} بنجاح.");
                    $successCount++;
                } else {
                    $this->error("فشل إنشاء قيد المحاسبة للإيداع رقم {$deposit->id}.");
                    $errorCount++;
                }

                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->error("حدث خطأ أثناء إنشاء قيد المحاسبة للإيداع رقم {$deposit->id}: " . $e->getMessage());
                Log::error("حدث خطأ أثناء إعادة إنشاء قيد المحاسبة للإيداع رقم {$deposit->id}: " . $e->getMessage());
                $errorCount++;
            }
        }

        $this->info("اكتملت العملية. تم إنشاء {$successCount} قيد محاسبي بنجاح. فشل إنشاء {$errorCount} قيد محاسبي.");

        return 0;
    }
}
