<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use App\Models\UserPermission;

class ListInventoryPermissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'list:inventory-permissions {email? : Email of the user to check}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'List all inventory permissions and check if a user has them';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Get all inventory permissions
        $permissions = Permission::where('slug', 'like', '%inventory%')
            ->orWhere('slug', 'like', '%warehouse%')
            ->orWhere('slug', 'like', '%supplier%')
            ->orWhere('slug', 'like', '%purchase%')
            ->get(['id', 'name', 'slug']);

        $this->info('Found ' . $permissions->count() . ' inventory permissions:');
        $this->table(['ID', 'Name', 'Slug'], $permissions->toArray());

        // Check if email is provided
        $email = $this->argument('email');
        if ($email) {
            $user = User::where('email', $email)->first();
            if (!$user) {
                $this->error("User with email {$email} not found.");
                return Command::FAILURE;
            }

            $this->info("Checking permissions for user: {$user->name} ({$user->email})");
            $this->info("User ID: {$user->id}");

            // Get user roles
            $roles = $user->roles;
            $this->info("User has " . $roles->count() . " roles:");
            $this->table(['ID', 'Name'], $roles->map(function ($role) {
                return ['ID' => $role->id, 'Name' => $role->name];
            })->toArray());

            // Check each permission
            $this->info("Checking each inventory permission:");
            $rows = [];
            foreach ($permissions as $permission) {
                $hasPermission = $user->hasPermission($permission->slug);
                $rows[] = [
                    'ID' => $permission->id,
                    'Slug' => $permission->slug,
                    'Has Permission' => $hasPermission ? 'Yes' : 'No'
                ];
            }
            $this->table(['ID', 'Slug', 'Has Permission'], $rows);

            // Get direct permissions
            $directPermissions = UserPermission::where('user_id', $user->id)
                ->whereIn('permission_id', $permissions->pluck('id'))
                ->get();

            if ($directPermissions->count() > 0) {
                $this->info("User has " . $directPermissions->count() . " direct inventory permissions:");
                $this->table(
                    ['Permission ID', 'Granted', 'Expires At'],
                    $directPermissions->map(function ($p) {
                        return [
                            'Permission ID' => $p->permission_id,
                            'Granted' => $p->granted ? 'Yes' : 'No',
                            'Expires At' => $p->expires_at ? $p->expires_at->format('Y-m-d H:i:s') : 'Never'
                        ];
                    })->toArray()
                );
            } else {
                $this->info("User has no direct inventory permissions.");
            }

            // Check role permissions
            $this->info("Checking role permissions:");
            foreach ($roles as $role) {
                $rolePermissions = $role->permissions()
                    ->whereIn('id', $permissions->pluck('id'))
                    ->get(['id', 'slug']);

                $this->info("Role: {$role->name} has " . $rolePermissions->count() . " inventory permissions:");
                if ($rolePermissions->count() > 0) {
                    $this->table(['ID', 'Slug'], $rolePermissions->toArray());
                } else {
                    $this->line("None");
                }
            }
        }

        return Command::SUCCESS;
    }
}
