<?php

namespace App\Console\Commands;

use App\Services\AutoInvoiceService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class GenerateAutoInvoices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoices:generate {--type=all : Type of invoices to generate (all, contracts, maintenance)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate automatic invoices from contracts and maintenance schedules';

    /**
     * The auto invoice service.
     *
     * @var \App\Services\AutoInvoiceService
     */
    protected $autoInvoiceService;

    /**
     * Create a new command instance.
     *
     * @param  \App\Services\AutoInvoiceService  $autoInvoiceService
     * @return void
     */
    public function __construct(AutoInvoiceService $autoInvoiceService)
    {
        parent::__construct();
        $this->autoInvoiceService = $autoInvoiceService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $type = $this->option('type');
        $this->info('Starting automatic invoice generation...');

        try {
            if ($type === 'all' || $type === 'contracts') {
                $this->info('Generating invoices from contracts...');
                $contractStats = $this->autoInvoiceService->generateContractInvoices();
                $this->displayStats('Contract Invoices', $contractStats);
            }

            if ($type === 'all' || $type === 'maintenance') {
                $this->info('Generating invoices from maintenance schedules...');
                $maintenanceStats = $this->autoInvoiceService->generateMaintenanceInvoices();
                $this->displayStats('Maintenance Invoices', $maintenanceStats);
            }

            $this->info('Automatic invoice generation completed successfully.');
            return 0;
        } catch (\Exception $e) {
            $this->error('Error generating invoices: ' . $e->getMessage());
            Log::error('Error in invoice generation command: ' . $e->getMessage());
            return 1;
        }
    }

    /**
     * Display statistics about generated invoices.
     *
     * @param  string  $title
     * @param  array  $stats
     * @return void
     */
    protected function displayStats($title, $stats)
    {
        $this->info("\n{$title} Statistics:");
        $this->info("Total processed: {$stats['total']}");
        $this->info("Successfully generated: {$stats['success']}");
        $this->info("Failed: {$stats['failed']}");

        if (!empty($stats['companies'])) {
            $this->info("\nBy Company:");
            $this->table(
                ['Company', 'Success', 'Failed'],
                collect($stats['companies'])->map(function ($company, $id) {
                    return [
                        $company['name'],
                        $company['success'],
                        $company['failed']
                    ];
                })->toArray()
            );
        }
    }
}
