<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CashTransaction;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\FiscalYear;
use App\Models\AccountingPeriod;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CreateJournalEntriesForCashTransactions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'fix:cash-transaction-entries';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create journal entries for existing cash transactions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('بدء إصلاح العلاقة بين معاملات الخزينة والقيود المحاسبية...');

        // Get all cash transactions
        $this->info('جاري البحث عن معاملات الخزينة...');
        $allTransactions = CashTransaction::count();
        $this->info("إجمالي معاملات الخزينة: {$allTransactions}");

        // Check if there are any journal entries
        $journalEntries = JournalEntry::count();
        $this->info("إجمالي القيود المحاسبية الحالية: {$journalEntries}");

        // Check if there are any cash transaction entries
        $cashTransactionEntries = DB::table('cash_transaction_entries')->count();
        $this->info("إجمالي علاقات معاملات الخزينة بالقيود المحاسبية: {$cashTransactionEntries}");

        // Get all cash transactions that don't have journal entries
        $transactions = CashTransaction::whereNotIn('id', function ($query) {
            $query->select('cash_transaction_id')
                ->from('cash_transaction_entries');
        })->get();

        $this->info("عدد معاملات الخزينة التي تحتاج إلى قيود محاسبية: {$transactions->count()}");

        if ($transactions->isEmpty()) {
            $this->info('لا توجد معاملات خزينة بدون قيود محاسبية.');
            return;
        }

        $this->info('تم العثور على ' . $transactions->count() . ' معاملة خزينة بدون قيود محاسبية.');

        // Get current fiscal year and accounting period
        $fiscalYear = FiscalYear::where('is_active', true)->first();
        if (!$fiscalYear) {
            $this->error('لا توجد سنة مالية نشطة. يرجى إنشاء واحدة أولاً.');
            return;
        }
        $this->info("السنة المالية النشطة: {$fiscalYear->name}");

        $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now())
            ->first();

        if (!$accountingPeriod) {
            // Try to get any accounting period for this fiscal year
            $accountingPeriod = AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)->first();
            if (!$accountingPeriod) {
                $this->error('لا توجد فترة محاسبية متاحة. يرجى إنشاء واحدة أولاً.');
                return;
            }
            $this->info("تم استخدام الفترة المحاسبية: {$accountingPeriod->name} (ليست الفترة الحالية)");
        } else {
            $this->info("الفترة المحاسبية الحالية: {$accountingPeriod->name}");
        }

        // Get cash account and revenue account
        $cashAccount = ChartOfAccount::where(function ($query) {
            $query->where('name', 'like', '%cash%')
                ->orWhere('name', 'like', '%الصندوق%')
                ->orWhere('name', 'like', '%خزينة%')
                ->orWhere('name_en', 'like', '%cash%')
                ->orWhere('is_cash_account', true)
                ->orWhere('account_type', 'cash');
        })
            ->first();

        if (!$cashAccount) {
            // إذا لم يتم العثور على حساب الصندوق، قم بإنشاء واحد
            $cashAccount = new ChartOfAccount();
            $cashAccount->company_id = 1; // استخدم الشركة الافتراضية
            $cashAccount->account_code = '1001';
            $cashAccount->name = 'الصندوق';
            $cashAccount->name_en = 'Cash';
            $cashAccount->account_type = 'cash';
            $cashAccount->is_active = true;
            $cashAccount->save();
            $this->info('تم إنشاء حساب الصندوق تلقائياً.');
        }

        $revenueAccount = ChartOfAccount::where(function ($query) {
            $query->where('name', 'like', '%revenue%')
                ->orWhere('name', 'like', '%إيراد%')
                ->orWhere('name', 'like', '%دخل%')
                ->orWhere('name_en', 'like', '%revenue%')
                ->orWhere('account_type', 'revenue');
        })
            ->first();

        if (!$revenueAccount) {
            // إذا لم يتم العثور على حساب الإيرادات، قم بإنشاء واحد
            $revenueAccount = new ChartOfAccount();
            $revenueAccount->company_id = 1; // استخدم الشركة الافتراضية
            $revenueAccount->account_code = '4001';
            $revenueAccount->name = 'إيرادات متنوعة';
            $revenueAccount->name_en = 'Miscellaneous Revenue';
            $revenueAccount->account_type = 'revenue';
            $revenueAccount->is_active = true;
            $revenueAccount->save();
            $this->info('تم إنشاء حساب الإيرادات تلقائياً.');
        }

        $bar = $this->output->createProgressBar($transactions->count());
        $bar->start();

        $createdCount = 0;

        foreach ($transactions as $transaction) {
            $this->info("معالجة معاملة الخزينة رقم: {$transaction->id}, النوع: {$transaction->type}, المبلغ: {$transaction->amount}");

            DB::beginTransaction();
            try {
                // Create journal entry
                $entry = new JournalEntry();
                $entry->company_id = $transaction->company_id ?? 1;
                $entry->fiscal_year_id = $fiscalYear->id;
                $entry->accounting_period_id = $accountingPeriod->id;
                $entry->entry_number = JournalEntry::generateEntryNumber($transaction->company_id ?? 1);
                $entry->entry_date = $transaction->created_at ?? now();
                $entry->reference_type = 'App\\Models\\CashTransaction';
                $entry->reference_id = $transaction->id;
                $entry->description = ($transaction->type == 'deposit' ? 'إيداع نقدي' : 'سحب نقدي') . ' - ' . ($transaction->notes ?? 'معاملة خزينة');
                $entry->created_by = $transaction->user_id ?? 1;
                $entry->is_posted = false;
                $entry->is_approved = false;
                $entry->save();

                $this->info("تم إنشاء القيد المحاسبي رقم: {$entry->id}, رقم القيد: {$entry->entry_number}");

                // Create journal entry items
                if ($transaction->type == 'deposit') {
                    // Debit cash account
                    $debitItem = new JournalEntryItem();
                    $debitItem->journal_entry_id = $entry->id;
                    $debitItem->account_id = $cashAccount->id;
                    $debitItem->debit = $transaction->amount;
                    $debitItem->credit = 0;
                    $debitItem->description = 'إيداع في الصندوق';
                    $debitItem->save();

                    // Credit revenue account
                    $creditItem = new JournalEntryItem();
                    $creditItem->journal_entry_id = $entry->id;
                    $creditItem->account_id = $revenueAccount->id;
                    $creditItem->debit = 0;
                    $creditItem->credit = $transaction->amount;
                    $creditItem->description = 'إيراد من معاملة خزينة';
                    $creditItem->save();
                } elseif ($transaction->type == 'withdrawal') {
                    // Credit cash account
                    $creditItem = new JournalEntryItem();
                    $creditItem->journal_entry_id = $entry->id;
                    $creditItem->account_id = $cashAccount->id;
                    $creditItem->debit = 0;
                    $creditItem->credit = $transaction->amount;
                    $creditItem->description = 'سحب من الصندوق';
                    $creditItem->save();

                    // Debit expense account (using revenue for now, should be replaced with proper expense account)
                    $debitItem = new JournalEntryItem();
                    $debitItem->journal_entry_id = $entry->id;
                    $debitItem->account_id = $revenueAccount->id;
                    $debitItem->debit = $transaction->amount;
                    $debitItem->credit = 0;
                    $debitItem->description = 'مصروف من معاملة خزينة';
                    $debitItem->save();
                }

                // Create record in cash_transaction_entries table
                DB::table('cash_transaction_entries')->insert([
                    'cash_transaction_id' => $transaction->id,
                    'journal_entry_id' => $entry->id,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                DB::commit();
                $createdCount++;
                $this->info("تم إنشاء القيد المحاسبي بنجاح لمعاملة الخزينة رقم: {$transaction->id}");
            } catch (\Exception $e) {
                DB::rollBack();
                $this->error('خطأ في إنشاء القيد المحاسبي لمعاملة الخزينة رقم ' . $transaction->id . ': ' . $e->getMessage());
                $this->error('تفاصيل الخطأ: ' . $e->getTraceAsString());
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine();
        $this->info('تم الانتهاء من الإصلاح. تم إصلاح ' . $createdCount . ' علاقة.');
    }
}
