<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\JournalEntry;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class CheckTrialBalance extends Command
{
    /**
     * اسم الأمر وتوصيفه
     *
     * @var string
     */
    protected $signature = 'accounting:check-trial-balance {--fix : تصحيح القيود غير المتوازنة تلقائيًا} {--email= : إرسال تقرير بالنتائج إلى البريد الإلكتروني المحدد} {--company= : معرف الشركة للتحقق من ميزان المراجعة الخاص بها فقط}';

    /**
     * وصف الأمر
     *
     * @var string
     */
    protected $description = 'فحص توازن ميزان المراجعة وتحديد القيود غير المتوازنة والحسابات غير المتوازنة';

    /**
     * تنفيذ الأمر
     *
     * @return int
     */
    public function handle()
    {
        $this->info('بدء فحص ميزان المراجعة...');

        $unbalancedEntries = [];
        $unbalancedAccounts = [];
        $fixedEntries = [];
        $updatedAccounts = [];

        // فحص توازن قيود اليومية
        $this->info('فحص توازن قيود اليومية...');

        // تطبيق عزل الشركات إذا تم تحديد معرف الشركة
        $companyId = $this->option('company');
        $entriesQuery = JournalEntry::with('items');

        if ($companyId) {
            $this->info("تطبيق عزل الشركات - فحص القيود الخاصة بالشركة رقم: {$companyId}");
            $entriesQuery->where('company_id', $companyId);
        }

        $entries = $entriesQuery->get();

        foreach ($entries as $entry) {
            if (!$entry->isBalanced()) {
                $difference = $entry->total_debit - $entry->total_credit;

                $unbalancedEntries[] = [
                    'id' => $entry->id,
                    'entry_number' => $entry->entry_number,
                    'entry_date' => $entry->entry_date,
                    'description' => $entry->description,
                    'total_debit' => $entry->total_debit,
                    'total_credit' => $entry->total_credit,
                    'difference' => $difference
                ];

                // تصحيح القيد إذا تم تحديد خيار التصحيح التلقائي
                if ($this->option('fix')) {
                    try {
                        // البحث عن حساب فروق ميزان المراجعة
                        $adjustmentAccount = ChartOfAccount::where('account_code', 'ADJ-DIFF')->first();

                        if (!$adjustmentAccount) {
                            $this->error('لم يتم العثور على حساب فروق ميزان المراجعة. يرجى إنشاء الحساب أولاً.');
                            continue;
                        }

                        // إنشاء بند تصحيح
                        $item = new \App\Models\JournalEntryItem();
                        $item->journal_entry_id = $entry->id;
                        $item->account_id = $adjustmentAccount->id;
                        $item->description = 'تصحيح فرق ميزان المراجعة';

                        if ($difference > 0) {
                            // إذا كان المدين أكبر، نضيف بند دائن
                            $item->debit = 0;
                            $item->credit = $difference;
                        } else {
                            // إذا كان الدائن أكبر، نضيف بند مدين
                            $item->debit = abs($difference);
                            $item->credit = 0;
                        }

                        $item->save();

                        $fixedEntries[] = [
                            'entry_number' => $entry->entry_number,
                            'difference' => $difference
                        ];

                        $this->info("تم تصحيح القيد رقم: {$entry->entry_number} (الفرق: {$difference})");

                        // تسجيل التصحيح في ملف السجل
                        Log::info("تم تصحيح قيد غير متوازن: {$entry->entry_number}", [
                            'entry_id' => $entry->id,
                            'difference' => $difference,
                            'adjustment_account' => $adjustmentAccount->account_code
                        ]);
                    } catch (\Exception $e) {
                        $this->error("خطأ في تصحيح القيد رقم {$entry->entry_number}: " . $e->getMessage());
                        Log::error("خطأ في تصحيح القيد رقم {$entry->entry_number}", [
                            'entry_id' => $entry->id,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
            }
        }

        // فحص أرصدة الحسابات
        $this->info('فحص أرصدة الحسابات...');

        // تطبيق عزل الشركات إذا تم تحديد معرف الشركة
        $accountsQuery = ChartOfAccount::query();

        if ($companyId) {
            $this->info("تطبيق عزل الشركات - فحص الحسابات الخاصة بالشركة رقم: {$companyId}");
            $accountsQuery->where('company_id', $companyId);
        }

        $accounts = $accountsQuery->get();

        foreach ($accounts as $account) {
            // حساب الرصيد من بنود القيود المحاسبية
            // تطبيق عزل الشركات على بنود القيود المحاسبية أيضًا
            $journalItems = $account->journalItems();

            if ($companyId) {
                $journalItems = $journalItems->join('journal_entries', 'journal_entry_items.journal_entry_id', '=', 'journal_entries.id')
                    ->where('journal_entries.company_id', $companyId);
                $debits = $journalItems->sum('journal_entry_items.debit');
                $credits = $journalItems->sum('journal_entry_items.credit');
            } else {
                $debits = $journalItems->sum('debit');
                $credits = $journalItems->sum('credit');
            }

            // حساب الرصيد المتوقع
            $expectedBalance = 0;
            if (in_array($account->account_type, ['asset', 'expense'])) {
                $expectedBalance = $account->opening_balance + $debits - $credits;
            } else {
                $expectedBalance = $account->opening_balance + $credits - $debits;
            }

            // مقارنة الرصيد المتوقع بالرصيد الحالي
            if (abs($expectedBalance - $account->current_balance) > 0.01) {
                $unbalancedAccounts[] = [
                    'id' => $account->id,
                    'name' => $account->name,
                    'account_code' => $account->account_code,
                    'current_balance' => $account->current_balance,
                    'expected_balance' => $expectedBalance,
                    'difference' => $expectedBalance - $account->current_balance
                ];

                // تحديث الرصيد إذا تم تحديد خيار التصحيح التلقائي
                if ($this->option('fix')) {
                    try {
                        $oldBalance = $account->current_balance;

                        // تحديث الرصيد
                        $account->updateBalance();

                        $updatedAccounts[] = [
                            'name' => $account->name,
                            'account_code' => $account->account_code,
                            'old_balance' => $oldBalance,
                            'new_balance' => $account->current_balance
                        ];

                        $this->info("تم تحديث حساب: {$account->name} (رمز: {$account->account_code})");

                        // تسجيل التحديث في ملف السجل
                        Log::info("تم تحديث رصيد حساب: {$account->name}", [
                            'account_id' => $account->id,
                            'old_balance' => $oldBalance,
                            'new_balance' => $account->current_balance
                        ]);
                    } catch (\Exception $e) {
                        $this->error("خطأ في تحديث حساب {$account->name}: " . $e->getMessage());
                        Log::error("خطأ في تحديث حساب {$account->name}", [
                            'account_id' => $account->id,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
            }
        }

        // عرض ملخص النتائج
        $this->info('');
        $this->info('ملخص النتائج:');
        $this->info('-------------');
        $this->info('عدد القيود غير المتوازنة: ' . count($unbalancedEntries));
        $this->info('عدد الحسابات غير المتوازنة: ' . count($unbalancedAccounts));

        if ($this->option('fix')) {
            $this->info('عدد القيود التي تم تصحيحها: ' . count($fixedEntries));
            $this->info('عدد الحسابات التي تم تحديثها: ' . count($updatedAccounts));
        }

        // إرسال تقرير بالنتائج إلى البريد الإلكتروني المحدد
        $email = $this->option('email');
        if ($email) {
            $this->info('جاري إرسال تقرير بالنتائج إلى البريد الإلكتروني: ' . $email);

            try {
                // إنشاء محتوى التقرير
                $report = [
                    'unbalanced_entries' => $unbalancedEntries,
                    'unbalanced_accounts' => $unbalancedAccounts,
                    'fixed_entries' => $fixedEntries,
                    'updated_accounts' => $updatedAccounts,
                    'summary' => [
                        'unbalanced_entries_count' => count($unbalancedEntries),
                        'unbalanced_accounts_count' => count($unbalancedAccounts),
                        'fixed_entries_count' => count($fixedEntries),
                        'updated_accounts_count' => count($updatedAccounts)
                    ]
                ];

                // إرسال التقرير
                Mail::raw(json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), function ($message) use ($email) {
                    $message->to($email)
                        ->subject('تقرير فحص ميزان المراجعة');
                });

                $this->info('تم إرسال التقرير بنجاح.');
            } catch (\Exception $e) {
                $this->error('خطأ في إرسال التقرير: ' . $e->getMessage());
                Log::error('خطأ في إرسال تقرير فحص ميزان المراجعة', [
                    'error' => $e->getMessage()
                ]);
            }
        }

        $this->info('تم الانتهاء من فحص ميزان المراجعة.');

        return 0;
    }
}
