<?php

namespace App\Console\Commands;

use App\Services\NotificationService;
use App\Services\ActivityLogService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckOverdueMaintenanceSchedules extends Command
{
    /**
     * اسم وتوقيع الأمر.
     *
     * @var string
     */
    protected $signature = 'maintenance:check-overdue';

    /**
     * وصف الأمر.
     *
     * @var string
     */
    protected $description = 'التحقق من جدولة الصيانة المتأخرة وإرسال إشعارات متقدمة';

    /**
     * خدمة الإشعارات.
     *
     * @var \App\Services\NotificationService
     */
    protected $notificationService;

    /**
     * خدمة تسجيل النشاط.
     *
     * @var \App\Services\ActivityLogService
     */
    protected $activityLogService;

    /**
     * إنشاء مثيل جديد من الأمر.
     *
     * @param  \App\Services\NotificationService  $notificationService
     * @param  \App\Services\ActivityLogService  $activityLogService
     * @return void
     */
    public function __construct(NotificationService $notificationService, ActivityLogService $activityLogService)
    {
        parent::__construct();
        $this->notificationService = $notificationService;
        $this->activityLogService = $activityLogService;
    }

    /**
     * تنفيذ الأمر.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('جاري التحقق من جدولة الصيانة المتأخرة...');

        try {
            // التحقق من الصيانة المتأخرة
            $overdueCount = $this->notificationService->checkOverdueMaintenanceSchedules();
            $this->info("تم إنشاء {$overdueCount} إشعار للصيانة المتأخرة.");

            // إرسال جميع الإشعارات المعلقة
            $sentCount = $this->notificationService->sendPendingNotifications();
            $this->info("تم إرسال {$sentCount} إشعار معلق.");

            $this->info('تم الانتهاء من التحقق من الصيانة المتأخرة بنجاح.');

            // تسجيل النشاط باستخدام خدمة ActivityLogService
            $this->activityLogService->log(
                'maintenance_check_overdue',
                'تم التحقق من الصيانة المتأخرة وإرسال الإشعارات',
                null,
                null,
                [
                    'overdue_count' => $overdueCount,
                    'sent_count' => $sentCount
                ]
            );

            return 0;
        } catch (\Exception $e) {
            $this->error('حدث خطأ: ' . $e->getMessage());
            Log::error('فشل التحقق من جدولة الصيانة المتأخرة: ' . $e->getMessage());
            return 1;
        }
    }
}
