<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\AutoPurchaseRule;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\User;
use Carbon\Carbon;

class CheckAutoPurchaseRules extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:check-auto-purchase';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check inventory levels and create automatic purchase orders based on rules';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('بدء فحص قواعد الشراء التلقائي...');

        // الحصول على القواعد النشطة التي يجب فحصها اليوم
        $rules = AutoPurchaseRule::where('is_active', true)
            ->where('next_check_date', '<=', Carbon::today())
            ->with(['item', 'supplier', 'warehouse', 'company'])
            ->get();

        $this->info("تم العثور على {$rules->count()} قاعدة للفحص.");

        $createdOrders = 0;
        $processedItems = 0;

        // تجميع القواعد حسب الشركة والمورد والمستودع
        $groupedRules = $rules->groupBy(function ($rule) {
            return $rule->company_id . '-' . $rule->supplier_id . '-' . $rule->warehouse_id;
        });

        foreach ($groupedRules as $group => $groupRules) {
            $firstRule = $groupRules->first();
            $company = $firstRule->company;
            $supplier = $firstRule->supplier;
            $warehouse = $firstRule->warehouse;

            $this->info("معالجة المجموعة: شركة {$company->name} - مورد {$supplier->name} - مستودع {$warehouse->name}");

            $itemsToOrder = [];

            foreach ($groupRules as $rule) {
                $processedItems++;

                // فحص المخزون الحالي
                $stock = $rule->item->stocks()
                    ->where('warehouse_id', $rule->warehouse_id)
                    ->first();

                $currentQuantity = $stock ? $stock->quantity - $stock->reserved_quantity : 0;

                $this->info("فحص العنصر: {$rule->item->name} - المخزون الحالي: {$currentQuantity} - الحد الأدنى: {$rule->min_quantity}");

                // إذا كان المخزون أقل من الحد الأدنى، أضف العنصر إلى قائمة الطلب
                if ($currentQuantity <= $rule->min_quantity) {
                    $this->info("العنصر {$rule->item->name} يحتاج إلى طلب - الكمية: {$rule->order_quantity}");

                    $itemsToOrder[] = [
                        'rule' => $rule,
                        'item' => $rule->item,
                        'quantity' => $rule->order_quantity,
                    ];
                }

                // تحديث تاريخ الفحص التالي
                if ($rule->frequency_days > 0) {
                    $rule->next_check_date = Carbon::today()->addDays($rule->frequency_days);
                } else {
                    $rule->next_check_date = Carbon::today()->addDay();
                }
                $rule->save();
            }

            // إذا كان هناك عناصر للطلب، أنشئ طلب شراء
            if (count($itemsToOrder) > 0) {
                $this->info("إنشاء طلب شراء جديد لـ {$supplier->name} مع " . count($itemsToOrder) . " عنصر");

                // الحصول على مستخدم نظام من الشركة لإنشاء الطلب باسمه
                $user = User::where('company_id', $company->id)
                    ->where('role', 'admin')
                    ->first();

                if (!$user) {
                    $this->error("لا يمكن العثور على مستخدم مناسب لإنشاء الطلب للشركة {$company->name}");
                    continue;
                }

                // إنشاء طلب شراء جديد
                $purchaseOrder = new PurchaseOrder();
                $purchaseOrder->company_id = $company->id;
                $purchaseOrder->supplier_id = $supplier->id;
                $purchaseOrder->warehouse_id = $warehouse->id;
                $purchaseOrder->created_by = $user->id;
                $purchaseOrder->po_number = 'AUTO-' . time();
                $purchaseOrder->order_date = Carbon::today();
                $purchaseOrder->expected_delivery_date = Carbon::today()->addDays(7);
                $purchaseOrder->status = 'draft';
                $purchaseOrder->notes = 'تم إنشاء هذا الطلب تلقائيًا بواسطة نظام الطلب التلقائي';

                // حساب المجاميع
                $subtotal = 0;

                foreach ($itemsToOrder as $orderItem) {
                    $item = $orderItem['item'];
                    $quantity = $orderItem['quantity'];
                    $subtotal += $item->purchase_price * $quantity;
                }

                $purchaseOrder->subtotal = $subtotal;
                $purchaseOrder->tax_amount = 0;
                $purchaseOrder->discount_amount = 0;
                $purchaseOrder->total_amount = $subtotal;

                $purchaseOrder->save();

                // إضافة عناصر الطلب
                foreach ($itemsToOrder as $orderItem) {
                    $item = $orderItem['item'];
                    $rule = $orderItem['rule'];
                    $quantity = $orderItem['quantity'];

                    $poItem = new PurchaseOrderItem();
                    $poItem->purchase_order_id = $purchaseOrder->id;
                    $poItem->item_id = $item->id;
                    $poItem->quantity = $quantity;
                    $poItem->received_quantity = 0;
                    $poItem->unit_price = $item->purchase_price;
                    $poItem->tax_rate = 0;
                    $poItem->discount_amount = 0;
                    $poItem->total_amount = $item->purchase_price * $quantity;
                    $poItem->notes = 'تم إنشاؤه تلقائيًا - الحد الأدنى: ' . $rule->min_quantity;
                    $poItem->save();

                    // تحديث وقت آخر تشغيل للقاعدة
                    $rule->last_triggered_at = now();
                    $rule->save();
                }

                $createdOrders++;
            }
        }

        $this->info("اكتمل الفحص. تمت معالجة {$processedItems} عنصر وإنشاء {$createdOrders} طلب شراء تلقائي.");

        return 0;
    }
}
