<?php

namespace App\Console\Commands;

use App\Services\NotificationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckAllMaintenanceSchedules extends Command
{
    /**
     * اسم وتوقيع الأمر.
     *
     * @var string
     */
    protected $signature = 'maintenance:check-all';

    /**
     * وصف الأمر.
     *
     * @var string
     */
    protected $description = 'التحقق من جميع أنواع جدولة الصيانة (اليومية، المستقبلية، المتأخرة) وإرسال الإشعارات';

    /**
     * خدمة الإشعارات.
     *
     * @var \App\Services\NotificationService
     */
    protected $notificationService;

    /**
     * إنشاء مثيل جديد من الأمر.
     *
     * @param  \App\Services\NotificationService  $notificationService
     * @return void
     */
    public function __construct(NotificationService $notificationService)
    {
        parent::__construct();
        $this->notificationService = $notificationService;
    }

    /**
     * تنفيذ الأمر.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('جاري التحقق من جميع أنواع جدولة الصيانة...');

        try {
            // التحقق من الصيانة المستحقة اليوم
            $todayCount = $this->notificationService->checkTodayMaintenanceSchedules();
            $this->info("تم إنشاء {$todayCount} إشعار للصيانة المستحقة اليوم.");

            // التحقق من الصيانة المستقبلية (7 أيام مقدماً)
            $upcomingCount = $this->notificationService->checkUpcomingMaintenanceSchedules(7);
            $this->info("تم إنشاء {$upcomingCount} إشعار للصيانة المستقبلية.");

            // التحقق من الصيانة المتأخرة
            $overdueCount = $this->notificationService->checkOverdueMaintenanceSchedules();
            $this->info("تم إنشاء {$overdueCount} إشعار للصيانة المتأخرة.");

            // التحقق من العقود التي ستنتهي قريباً (30 يوم مقدماً)
            $contractCount = $this->notificationService->checkExpiringContracts(30);
            $this->info("تم إنشاء {$contractCount} إشعار للعقود التي ستنتهي قريباً.");

            // إرسال جميع الإشعارات المعلقة
            $sentCount = $this->notificationService->sendPendingNotifications();
            $this->info("تم إرسال {$sentCount} إشعار معلق.");

            $totalCount = $todayCount + $upcomingCount + $overdueCount + $contractCount;
            $this->info("تم الانتهاء من التحقق من جميع أنواع الصيانة بنجاح. إجمالي الإشعارات: {$totalCount}");

            // تسجيل النشاط
            activity()
                ->withProperties([
                    'today_count' => $todayCount,
                    'upcoming_count' => $upcomingCount,
                    'overdue_count' => $overdueCount,
                    'contract_count' => $contractCount,
                    'sent_count' => $sentCount,
                    'total_count' => $totalCount
                ])
                ->log('تم التحقق من جميع أنواع الصيانة وإرسال الإشعارات');

            return 0;
        } catch (\Exception $e) {
            $this->error('حدث خطأ: ' . $e->getMessage());
            Log::error('فشل التحقق من جميع أنواع جدولة الصيانة: ' . $e->getMessage());
            return 1;
        }
    }
}
