<?php

namespace App\Console\Commands;

use App\Models\AutoPurchaseRule;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AutoPurchaseCheck extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:auto-purchase-check {--company=} {--user=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'فحص قواعد الشراء التلقائي وإنشاء طلبات الشراء عند الحاجة';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('بدء فحص قواعد الشراء التلقائي...');

        $companyId = $this->option('company');
        $userId = $this->option('user');

        // إذا تم تحديد شركة معينة
        if ($companyId) {
            $this->processCompanyRules($companyId, $userId);
        } else {
            // معالجة جميع الشركات
            $companies = DB::table('companies')->where('is_active', true)->pluck('id');

            foreach ($companies as $company) {
                $this->processCompanyRules($company, $userId);
            }
        }

        $this->info('تم الانتهاء من فحص قواعد الشراء التلقائي بنجاح.');
        return Command::SUCCESS;
    }

    /**
     * معالجة قواعد الشراء التلقائي لشركة معينة.
     *
     * @param int $companyId
     * @param int|null $userId
     * @return void
     */
    private function processCompanyRules($companyId, $userId = null)
    {
        $this->info("معالجة قواعد الشراء التلقائي للشركة رقم {$companyId}...");

        // تحديد المستخدم الذي سيتم استخدامه لإنشاء طلبات الشراء
        $user = null;
        if ($userId) {
            $user = User::find($userId);
        }

        if (!$user) {
            // البحث عن مستخدم مدير في الشركة
            $user = User::where('company_id', $companyId)
                ->whereHas('roles', function ($query) {
                    $query->where('name', 'admin')
                        ->orWhere('name', 'super_admin')
                        ->orWhere('name', 'inventory_manager');
                })
                ->where('is_active', true)
                ->first();
        }

        if (!$user) {
            $this->error("لم يتم العثور على مستخدم مناسب للشركة رقم {$companyId}");
            return;
        }

        // الحصول على القواعد النشطة التي يجب فحصها
        $rules = AutoPurchaseRule::where('company_id', $companyId)
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('next_check_date')
                    ->orWhere('next_check_date', '<=', now());
            })
            ->get();

        $this->info("تم العثور على {$rules->count()} قاعدة للفحص.");

        $createdOrders = 0;

        foreach ($rules as $rule) {
            // الحصول على مستوى المخزون الحالي
            $currentStock = InventoryStock::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->sum('quantity');

            $item = $rule->item;
            $this->line("فحص العنصر: {$item->name} (المخزون الحالي: {$currentStock}, الحد الأدنى: {$rule->min_quantity})");

            // التحقق مما إذا كان المخزون أقل من الحد الأدنى
            if ($currentStock <= $rule->min_quantity) {
                $this->warn("المخزون أقل من الحد الأدنى! إنشاء طلب شراء...");

                // إنشاء طلب شراء
                DB::beginTransaction();
                try {
                    $purchaseOrder = new PurchaseOrder();
                    $purchaseOrder->company_id = $companyId;
                    $purchaseOrder->supplier_id = $rule->supplier_id;
                    $purchaseOrder->warehouse_id = $rule->warehouse_id;
                    $purchaseOrder->auto_purchase_rule_id = $rule->id;
                    $purchaseOrder->is_auto_generated = true;
                    $purchaseOrder->order_date = now();
                    $purchaseOrder->expected_delivery_date = now()->addDays(7);
                    $purchaseOrder->status = 'pending';
                    $purchaseOrder->created_by = $user->id;
                    $purchaseOrder->notes = 'تم إنشاء هذا الطلب تلقائياً بواسطة نظام الشراء التلقائي';
                    $purchaseOrder->save();

                    // إضافة عنصر الطلب
                    $orderItem = new PurchaseOrderItem();
                    $orderItem->purchase_order_id = $purchaseOrder->id;
                    $orderItem->item_id = $rule->item_id;
                    $orderItem->quantity = $rule->order_quantity;
                    $orderItem->unit_price = $item->last_purchase_price ?? 0;
                    $orderItem->total_price = $orderItem->quantity * $orderItem->unit_price;
                    $orderItem->save();

                    // تحديث إجمالي الطلب
                    $purchaseOrder->total_amount = $orderItem->total_price;
                    $purchaseOrder->save();

                    // تحديث القاعدة
                    $rule->last_triggered_at = now();
                    $rule->next_check_date = now()->addDays($rule->frequency_days ?: 1);
                    $rule->save();

                    DB::commit();
                    $createdOrders++;

                    $this->info("تم إنشاء طلب الشراء رقم {$purchaseOrder->id} بنجاح.");
                } catch (\Exception $e) {
                    DB::rollBack();
                    $this->error("خطأ في إنشاء طلب شراء تلقائي: " . $e->getMessage());
                    Log::error('خطأ في إنشاء طلب شراء تلقائي: ' . $e->getMessage());
                }
            } else {
                $this->line("المخزون كافٍ، تحديث تاريخ الفحص التالي فقط.");
                // تحديث تاريخ الفحص التالي فقط
                $rule->next_check_date = now()->addDays($rule->frequency_days ?: 1);
                $rule->save();
            }
        }

        $this->info("تم الانتهاء من معالجة قواعد الشركة رقم {$companyId}. تم إنشاء {$createdOrders} طلب شراء.");
    }
}
