<?php

namespace App\Console\Commands;

use App\Models\Role;
use App\Models\User;
use App\Models\Permission;
use Illuminate\Console\Command;

class AssignProjectManagerRole extends Command
{
    /**
     * اسم الأمر وتوصيفه
     *
     * @var string
     */
    protected $signature = 'role:assign-project-manager {email? : البريد الإلكتروني للمستخدم}';

    /**
     * وصف الأمر
     *
     * @var string
     */
    protected $description = 'إنشاء دور مدير المشاريع وتعيينه للمستخدم المحدد';

    /**
     * تنفيذ الأمر
     *
     * @return int
     */
    public function handle()
    {
        // إنشاء دور مدير المشاريع إذا لم يكن موجودًا
        $projectManagerRole = Role::firstOrCreate(
            ['slug' => 'project_manager'],
            [
                'name' => 'مدير مشاريع',
                'description' => 'مسؤول عن إدارة مشاريع التركيب',
                'is_active' => true,
                'is_system_role' => true,
                'is_default' => false,
                'order' => 4,
            ]
        );

        $this->info('تم إنشاء دور مدير المشاريع بنجاح.');

        // إضافة الصلاحيات المناسبة للدور
        $projectManagerPermissions = [
            'view_installation_projects',
            'edit_installation_project',
            'manage_installation_projects',
            'view_project_elevators',
            'add_project_elevator',
            'edit_project_elevator',
            'view_project_phases',
            'add_project_phase',
            'edit_project_phase',
            'view_project_tasks',
            'add_project_task',
            'edit_project_task',
            'assign_project_task',
            'view_project_team',
            'add_project_team_member',
            'edit_project_team_member',
            'view_project_documents',
            'add_project_document',
            'edit_project_document',
            'view_project_expenses',
            'add_project_expense',
            'view_project_payments',
            'view_project_issues',
            'add_project_issue',
            'edit_project_issue',
            'resolve_project_issue',
            'view_project_milestones',
            'add_project_milestone',
            'edit_project_milestone',
            'complete_project_milestone',
            'view_project_checklists',
            'add_project_checklist',
            'edit_project_checklist',
            'complete_project_checklist',
        ];

        // الحصول على الصلاحيات من قاعدة البيانات
        $permissions = Permission::whereIn('slug', $projectManagerPermissions)
            ->orWhereIn('name', $projectManagerPermissions)
            ->get();

        // إضافة الصلاحيات للدور
        if ($permissions->count() > 0) {
            $projectManagerRole->permissions()->syncWithoutDetaching($permissions->pluck('id')->toArray());
            $this->info('تم إضافة ' . $permissions->count() . ' صلاحية لدور مدير المشاريع.');
        } else {
            $this->warn('لم يتم العثور على أي صلاحيات لإضافتها لدور مدير المشاريع.');

            // إضافة صلاحيات أساسية على الأقل
            $basicPermissions = Permission::where('name', 'like', '%project%')
                ->orWhere('slug', 'like', '%project%')
                ->orWhere('name', 'like', '%installation%')
                ->orWhere('slug', 'like', '%installation%')
                ->get();

            if ($basicPermissions->count() > 0) {
                $projectManagerRole->permissions()->syncWithoutDetaching($basicPermissions->pluck('id')->toArray());
                $this->info('تم إضافة ' . $basicPermissions->count() . ' صلاحية أساسية لدور مدير المشاريع.');
            } else {
                $this->warn('لم يتم العثور على أي صلاحيات أساسية لإضافتها لدور مدير المشاريع.');
            }
        }

        // تعيين الدور للمستخدم المحدد
        $email = $this->argument('email');

        if ($email) {
            // البحث عن المستخدم بالبريد الإلكتروني
            $user = User::where('email', $email)->first();

            if ($user) {
                $user->roles()->syncWithoutDetaching([$projectManagerRole->id]);
                $this->info('تم تعيين دور مدير المشاريع للمستخدم ' . $user->name . ' بنجاح.');
            } else {
                $this->error('لم يتم العثور على المستخدم بالبريد الإلكتروني: ' . $email);
            }
        } else {
            // إذا لم يتم تحديد بريد إلكتروني، نعرض قائمة بالمستخدمين للاختيار
            $users = User::where('is_active', true)->get(['id', 'name', 'email']);

            if ($users->isEmpty()) {
                $this->error('لا يوجد مستخدمين نشطين في النظام.');
                return 1;
            }

            $this->info('المستخدمون النشطون في النظام:');
            $headers = ['ID', 'الاسم', 'البريد الإلكتروني'];
            $rows = [];

            foreach ($users as $user) {
                $rows[] = [$user->id, $user->name, $user->email];
            }

            $this->table($headers, $rows);

            $userId = $this->ask('أدخل رقم المستخدم (ID) الذي تريد تعيين دور مدير المشاريع له:');
            $user = User::find($userId);

            if ($user) {
                $user->roles()->syncWithoutDetaching([$projectManagerRole->id]);
                $this->info('تم تعيين دور مدير المشاريع للمستخدم ' . $user->name . ' بنجاح.');
            } else {
                $this->error('لم يتم العثور على المستخدم برقم: ' . $userId);
                return 1;
            }
        }

        return 0;
    }
}
