<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Permission;
use App\Models\Role;
use App\Models\User;

class AssignCreateUsersPermission extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'assign:create-users-permission {email?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Assign create_users permission to a user or all users with create_user permission';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $email = $this->argument('email');
        $createUsersPermission = Permission::where('slug', 'create_users')->first();

        if (!$createUsersPermission) {
            $this->error("Permission 'create_users' not found. Please run the migration first.");
            return 1;
        }

        if ($email) {
            // Assign to specific user
            $user = User::where('email', $email)->first();

            if (!$user) {
                $this->error("User with email {$email} not found");
                return 1;
            }

            $this->assignPermissionToUser($user, $createUsersPermission);
        } else {
            // Assign to all users with create_user permission
            $createUserPermission = Permission::where('slug', 'create_user')->first();

            if (!$createUserPermission) {
                $this->error("Permission 'create_user' not found");
                return 1;
            }

            $users = User::whereHas('roles.permissions', function ($query) use ($createUserPermission) {
                $query->where('permissions.id', $createUserPermission->id);
            })->get();

            $this->info("Found " . $users->count() . " users with 'create_user' permission");

            foreach ($users as $user) {
                $this->assignPermissionToUser($user, $createUsersPermission);
            }

            // Also assign to all admin roles
            $adminRoles = Role::where('slug', 'admin')->orWhere('slug', 'super_admin')->get();
            foreach ($adminRoles as $role) {
                $role->permissions()->syncWithoutDetaching([$createUsersPermission->id]);
                $this->info("Assigned 'create_users' permission to role: {$role->name}");
            }
        }

        return 0;
    }

    /**
     * Assign permission to user
     */
    private function assignPermissionToUser($user, $permission)
    {
        if ($user->hasPermissionTo('create_users')) {
            $this->info("User {$user->name} already has 'create_users' permission");
            return;
        }

        // استخدام طريقة مباشرة بدلاً من grantPermission لتجنب تسجيل النشاط
        $permissionId = $permission->id;
        \DB::table('user_permissions')->updateOrInsert(
            [
                'user_id' => $user->id,
                'permission_id' => $permissionId,
            ],
            [
                'granted' => true,
                'source' => 'direct',
                'expires_at' => null,
                'created_at' => now(),
                'updated_at' => now(),
            ]
        );

        $this->info("Assigned 'create_users' permission to user: {$user->name} ({$user->email})");
    }
}
