# تحسينات ميزان المراجعة

تم تنفيذ مجموعة من التحسينات لمنع مشكلة عدم توازن ميزان المراجعة في النظام. هذه التحسينات تضمن دقة البيانات المالية وتوازن ميزان المراجعة في جميع الأوقات.

## 1. التحقق التلقائي من توازن القيود

تم تحسين نموذج `JournalEntry` لإضافة تحقق تلقائي من توازن القيد قبل حفظه. إذا كان هناك فرق صغير (أقل من 1)، يتم تصحيحه تلقائيًا باستخدام حساب فروق ميزان المراجعة.

```php
protected static function boot()
{
    parent::boot();
    
    // التحقق من توازن القيد قبل الحفظ
    static::saving(function ($entry) {
        // تجاهل التحقق إذا كان القيد جديدًا (لم يتم حفظه بعد)
        if (!$entry->exists) {
            return true;
        }
        
        // التحقق من توازن القيد
        if (!$entry->isBalanced()) {
            // تسجيل الخطأ في ملف السجل
            \Illuminate\Support\Facades\Log::warning('محاولة حفظ قيد غير متوازن: ' . $entry->entry_number);
            
            // إذا كان الفرق صغيرًا (أقل من 1)، نقوم بتصحيحه تلقائيًا
            $difference = $entry->total_debit - $entry->total_credit;
            if (abs($difference) < 1) {
                // البحث عن حساب فروق ميزان المراجعة
                $adjustmentAccount = \App\Models\ChartOfAccount::where('account_code', 'ADJ-DIFF')->first();
                
                if ($adjustmentAccount) {
                    // إنشاء بند تصحيح
                    $item = new \App\Models\JournalEntryItem();
                    $item->journal_entry_id = $entry->id;
                    $item->account_id = $adjustmentAccount->id;
                    $item->description = 'تصحيح تلقائي لفرق ميزان المراجعة';
                    
                    if ($difference > 0) {
                        // إذا كان المدين أكبر، نضيف بند دائن
                        $item->debit = 0;
                        $item->credit = $difference;
                    } else {
                        // إذا كان الدائن أكبر، نضيف بند مدين
                        $item->debit = abs($difference);
                        $item->credit = 0;
                    }
                    
                    $item->save();
                    
                    // تسجيل التصحيح في ملف السجل
                    \Illuminate\Support\Facades\Log::info('تم تصحيح قيد غير متوازن تلقائيًا: ' . $entry->entry_number);
                    
                    return true;
                }
            }
            
            // إذا كان الفرق كبيرًا، نمنع الحفظ
            if (abs($difference) >= 1) {
                throw new \Exception('لا يمكن حفظ القيد المحاسبي لعدم توازن المدين والدائن. الفرق: ' . $difference);
            }
        }
        
        return true;
    });
}
```

## 2. تحسين واجهة إدخال القيود المحاسبية

تم تحسين متحكم `JournalEntryController` لإضافة تحقق من توازن القيد قبل حفظه وتصحيحه تلقائيًا إذا كان الفرق صغيرًا.

```php
private function checkAndFixJournalEntryBalance(Request $request)
{
    // حساب إجمالي المدين والدائن
    $totalDebit = 0;
    $totalCredit = 0;

    foreach ($request->items as $item) {
        $totalDebit += floatval($item['debit'] ?? 0);
        $totalCredit += floatval($item['credit'] ?? 0);
    }

    $difference = abs($totalDebit - $totalCredit);
    
    // إذا كان الفرق أقل من 0.01، نعتبر القيد متوازنًا
    if ($difference <= 0.01) {
        return null;
    }
    
    // إذا كان الفرق صغيرًا (أقل من 1)، نحاول تصحيحه تلقائيًا
    if ($difference < 1) {
        // البحث عن حساب فروق ميزان المراجعة
        $adjustmentAccount = ChartOfAccount::where('account_code', 'ADJ-DIFF')->first();
        
        if ($adjustmentAccount) {
            // إضافة بند تصحيح إلى القيد
            $adjustmentItem = [
                'account_id' => $adjustmentAccount->id,
                'description' => 'تصحيح تلقائي لفرق ميزان المراجعة'
            ];
            
            if ($totalDebit > $totalCredit) {
                // إذا كان المدين أكبر، نضيف بند دائن
                $adjustmentItem['debit'] = 0;
                $adjustmentItem['credit'] = $difference;
            } else {
                // إذا كان الدائن أكبر، نضيف بند مدين
                $adjustmentItem['debit'] = $difference;
                $adjustmentItem['credit'] = 0;
            }
            
            // إضافة البند إلى القيد
            $request->merge(['items' => array_merge($request->items, [$adjustmentItem])]);
            
            // تسجيل التصحيح في ملف السجل
            \Illuminate\Support\Facades\Log::info('تم تصحيح قيد غير متوازن تلقائيًا عند الإنشاء');
            
            // إظهار رسالة للمستخدم
            session()->flash('warning', 'تم تصحيح فرق بسيط في القيد المحاسبي تلقائيًا (' . $difference . ') باستخدام حساب فروق ميزان المراجعة');
            
            return null;
        }
    }
    
    // إذا كان الفرق كبيرًا، نطلب من المستخدم تصحيحه يدويًا
    return [
        'error' => 'القيد المحاسبي غير متوازن. مجموع المدين: ' . $totalDebit . '، مجموع الدائن: ' . $totalCredit . '. الفرق: ' . $difference,
        'redirect' => 'back'
    ];
}
```

## 3. إضافة فحص دوري لميزان المراجعة

تم إنشاء أمر `CheckTrialBalance` للفحص الدوري لميزان المراجعة وتصحيح القيود غير المتوازنة تلقائيًا.

```php
php artisan accounting:check-trial-balance --fix --email=admin@example.com
```

تم إضافة الأمر إلى جدول المهام الدورية في `Kernel.php`:

```php
// فحص ميزان المراجعة أسبوعياً يوم السبت في الساعة 23:00 مساءً
$schedule->command('accounting:check-trial-balance --fix --email=admin@example.com')->weekly()->saturdays()->at('23:00');

// فحص ميزان المراجعة في نهاية كل شهر
$schedule->command('accounting:check-trial-balance --fix --email=admin@example.com')->monthlyOn(28, '23:30');
```

## 4. إضافة فحص عند تسجيل الدخول

تم إنشاء مستمع `CheckTrialBalanceOnLogin` للتحقق من توازن ميزان المراجعة عند تسجيل دخول المستخدمين الذين لديهم صلاحية إدارة الحسابات.

```php
public function handle(Login $event)
{
    // التحقق فقط للمستخدمين الذين لديهم صلاحية إدارة الحسابات
    if ($event->user->can('manage_accounting')) {
        try {
            // فحص توازن قيود اليومية
            $unbalancedEntries = $this->checkJournalEntriesBalance();
            
            // فحص أرصدة الحسابات
            $unbalancedAccounts = $this->checkAccountsBalance();
            
            // إذا كان هناك قيود غير متوازنة أو حسابات غير متوازنة، نعرض رسالة للمستخدم
            if (count($unbalancedEntries) > 0 || count($unbalancedAccounts) > 0) {
                $message = 'تم اكتشاف مشكلة في ميزان المراجعة: ';
                
                if (count($unbalancedEntries) > 0) {
                    $message .= count($unbalancedEntries) . ' قيد غير متوازن';
                }
                
                if (count($unbalancedEntries) > 0 && count($unbalancedAccounts) > 0) {
                    $message .= ' و ';
                }
                
                if (count($unbalancedAccounts) > 0) {
                    $message .= count($unbalancedAccounts) . ' حساب غير متوازن';
                }
                
                $message .= '. يرجى تشغيل أداة فحص ميزان المراجعة لتصحيح المشكلة.';
                
                Session::flash('warning', $message);
            }
        } catch (\Exception $e) {
            // تسجيل الخطأ في ملف السجل
            Log::error('خطأ في فحص ميزان المراجعة عند تسجيل الدخول', [
                'user_id' => $event->user->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
```

## 5. إنشاء حساب فروق ميزان المراجعة

تم إنشاء حساب خاص لتسوية فروق ميزان المراجعة:

```php
$adjustmentAccount = new ChartOfAccount();
$adjustmentAccount->name = 'فروق ميزان المراجعة';
$adjustmentAccount->account_code = 'ADJ-DIFF';
$adjustmentAccount->account_type = 'expense'; // نوع الحساب مصروفات
$adjustmentAccount->parent_id = null; // حساب رئيسي
$adjustmentAccount->opening_balance = 0;
$adjustmentAccount->current_balance = 0;
$adjustmentAccount->balance_type = 'debit'; // رصيد مدين
$adjustmentAccount->description = 'حساب لتسوية فروق ميزان المراجعة';
$adjustmentAccount->company_id = 1; // استخدام الشركة رقم 1
$adjustmentAccount->save();
```

## 6. سكريبتات تصحيح المشكلات

تم إنشاء ثلاثة سكريبتات لتصحيح المشكلات الموجودة:

1. **check_journal_entries.php**: لفحص القيود المحاسبية والحسابات غير المتوازنة.
2. **fix_unbalanced_entries.php**: لتصحيح القيود غير المتوازنة تلقائيًا.
3. **update_account_balances.php**: لتحديث أرصدة الحسابات بناءً على القيود المحاسبية.

## كيفية استخدام التحسينات

1. **تشغيل الفحص الدوري**: يتم تشغيل الفحص الدوري تلقائيًا حسب الجدول المحدد في `Kernel.php`.

2. **تشغيل الفحص يدويًا**: يمكن تشغيل الفحص يدويًا باستخدام الأمر التالي:
   ```
   php artisan accounting:check-trial-balance --fix
   ```

3. **تصحيح المشكلات الموجودة**: يمكن تصحيح المشكلات الموجودة باستخدام السكريبتات التالية:
   ```
   php check_journal_entries.php
   php fix_unbalanced_entries.php
   php update_account_balances.php
   ```

4. **التحقق من توازن ميزان المراجعة**: يمكن التحقق من توازن ميزان المراجعة من خلال صفحة ميزان المراجعة في النظام.

## ملاحظات هامة

1. يجب التأكد من وجود حساب فروق ميزان المراجعة برمز `ADJ-DIFF` قبل استخدام هذه التحسينات.

2. يجب مراجعة حساب فروق ميزان المراجعة بشكل دوري للتأكد من أن الفروق المسجلة فيه منطقية ومبررة.

3. إذا كانت هناك فروق كبيرة في ميزان المراجعة، يجب مراجعة القيود المحاسبية يدويًا وتصحيحها بدلاً من الاعتماد على التصحيح التلقائي.