# إصلاح قيد الفرادة في جدول دليل الحسابات

## 📋 المشكلة

عند محاولة إنشاء الحسابات الأساسية التلقائية لشركة جديدة، كان النظام يعطي خطأ:

```
SQLSTATE[23000]: Integrity constraint violation: 1062 Duplicate entry '1000' 
for key 'chart_of_accounts_account_code_unique'
```

### 🔍 السبب

كان هناك قيد فريد (unique constraint) على حقل `account_code` **فقط**، بدلاً من أن يكون على مجموعة `['company_id', 'account_code']` معاً.

هذا يعني أن:
- ❌ **الخطأ**: لا يمكن لشركتين مختلفتين استخدام نفس رمز الحساب (مثل 1000)
- ✅ **الصحيح**: كل شركة يجب أن يكون لها أكواد حسابات فريدة خاصة بها

## 🔧 الحل

تم إنشاء migration جديد لإصلاح المشكلة:

### الملف: `2025_10_01_213112_fix_chart_of_accounts_unique_constraint.php`

```php
public function up()
{
    Schema::table('chart_of_accounts', function (Blueprint $table) {
        // حذف القيد الفريد القديم على account_code فقط
        $table->dropUnique('chart_of_accounts_account_code_unique');
        
        // إضافة قيد فريد جديد على company_id و account_code معاً
        $table->unique(['company_id', 'account_code'], 'chart_of_accounts_company_account_unique');
    });
}
```

## ✅ النتيجة

بعد تطبيق الإصلاح:

### القيود الفريدة الحالية:
1. ✅ `PRIMARY` على `id`
2. ✅ `chart_of_accounts_company_account_unique` على `['company_id', 'account_code']`

### الفوائد:
- ✅ كل شركة يمكنها استخدام نفس أكواد الحسابات (1000, 2000, 3000, إلخ)
- ✅ عزل كامل بين الشركات في نظام SaaS
- ✅ لا يمكن تكرار رمز الحساب داخل نفس الشركة
- ✅ النظام التلقائي لإنشاء الحسابات الأساسية يعمل بشكل صحيح

## 📊 مثال

### قبل الإصلاح ❌
```
الشركة 1: حساب 1000 - الأصول ✓
الشركة 2: حساب 1000 - الأصول ✗ (خطأ: رمز مكرر)
```

### بعد الإصلاح ✅
```
الشركة 1: حساب 1000 - الأصول ✓
الشركة 2: حساب 1000 - الأصول ✓
الشركة 1: حساب 1000 - حساب آخر ✗ (خطأ: رمز مكرر في نفس الشركة)
```

## 🚀 التطبيق

### تشغيل الـ Migration:
```bash
php artisan migrate --path=database/migrations/2025_10_01_213112_fix_chart_of_accounts_unique_constraint.php
```

### التحقق من القيود:
```bash
php artisan tinker --execute="echo json_encode(DB::select('SHOW INDEX FROM chart_of_accounts'));"
```

## 📝 ملاحظات مهمة

1. **التوافق مع SaaS**: هذا الإصلاح ضروري لأي نظام SaaS متعدد الشركات
2. **البيانات الموجودة**: لا يؤثر على البيانات الموجودة، فقط يغير القيود
3. **الأمان**: يضمن عزل كامل بين بيانات الشركات المختلفة
4. **الأداء**: لا يؤثر على الأداء، بل قد يحسنه بسبب الفهرس المركب

## 🔄 التراجع (Rollback)

إذا احتجت للتراجع عن التغيير:

```bash
php artisan migrate:rollback --step=1
```

سيتم:
- حذف القيد المركب `chart_of_accounts_company_account_unique`
- إعادة القيد القديم `chart_of_accounts_account_code_unique`

⚠️ **تحذير**: التراجع سيعيد المشكلة الأصلية!

## 📚 المراجع

- [Laravel Unique Constraints](https://laravel.com/docs/migrations#unique-constraints)
- [Multi-Tenancy Best Practices](https://laravel.com/docs/eloquent#multi-tenancy)
- [Database Indexing](https://dev.mysql.com/doc/refman/8.0/en/create-index.html)

---

**تاريخ الإصلاح**: 2025-10-01  
**الإصدار**: 1.0  
**الحالة**: ✅ تم التطبيق بنجاح