---
description: دليل المطور لنظام إدارة شركات المصاعد
alwaysApply: true
---
هام جدا لا تحذف ابدا قاعدة البيانات او الشغل السابق قبل الرجوع الي مدير المشروع 

# دليل المطور لنظام إدارة شركات المصاعد (Elevators ERP)

## مقدمة

هذا الدليل مخصص للمطورين الجدد الذين ينضمون إلى مشروع نظام إدارة شركات المصاعد. يهدف هذا الدليل إلى تسهيل عملية التعرف على المشروع وفهم هيكله وآلية عمله، بالإضافة إلى توفير إرشادات للتطوير والاختبار.

## متطلبات التطوير

### البيئة المطلوبة
- PHP 8.0.2 أو أعلى
- MySQL 5.7 أو أعلى
- Composer 2.0 أو أعلى
- Node.js 14.0 أو أعلى
- NPM 6.0 أو أعلى

### الإعداد الأولي
1. استنساخ المستودع:
   ```bash
   git clone https://github.com/your-organization/elevators_erp.git
   cd elevators_erp
   ```

2. تثبيت التبعيات:
   ```bash
   composer install
   npm install
   ```

3. إعداد ملف البيئة:
   ```bash
   cp .env.example .env
   php artisan key:generate
   ```

4. تكوين قاعدة البيانات في ملف `.env`:
   ```
   DB_CONNECTION=mysql
   DB_HOST=127.0.0.1
   DB_PORT=3306
   DB_DATABASE=elevators_management
   DB_USERNAME=root
   DB_PASSWORD=
   ```

5. تشغيل الهجرات والبذور:
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

6. إنشاء رابط رمزي للتخزين:
   ```bash
   php artisan storage:link
   ```

7. تشغيل الخادم المحلي:
   ```bash
   php artisan serve
   ```

## هيكل المشروع

### المجلدات الرئيسية

```
elevators_erp/
├── app/                  # الكود الأساسي للتطبيق
│   ├── Models/           # نماذج البيانات
│   ├── Http/             # وحدات التحكم والوسائط
│   │   ├── Controllers/  # وحدات التحكم
│   │   ├── Middleware/   # وسائط البرمجة
│   │   └── Requests/     # طلبات النماذج والتحقق
│   └── Providers/        # مزودي الخدمات
├── config/               # ملفات الإعدادات
├── database/             # الهجرات والبذور
│   ├── migrations/       # ملفات الهجرة
│   └── seeders/          # ملفات البذور
├── public/               # الملفات العامة
├── resources/            # الموارد
│   ├── views/            # ملفات العرض
│   ├── js/               # ملفات JavaScript
│   └── css/              # ملفات CSS
├── routes/               # تعريفات المسارات
│   ├── web.php           # مسارات الويب
│   └── api.php           # مسارات API
└── tests/                # اختبارات التطبيق
```

## نظام متعدد المستأجرين (Multi-tenancy)

### كيفية العمل

يعتمد النظام على نهج "عزل البيانات" (Data Isolation) حيث يتم تخزين بيانات جميع الشركات في نفس قاعدة البيانات، ولكن يتم فصلها منطقيًا باستخدام حقل `company_id` في جميع الجداول ذات الصلة.

### آلية تنفيذ العزل

1. **الوسيط (Middleware)**: يتم استخدام وسيط `CheckCompany` للتحقق من أن المستخدم يمكنه الوصول فقط إلى بيانات الشركة التي ينتمي إليها.

2. **نطاق عام (Global Scope)**: يتم تطبيق نطاق عام على النماذج لتقييد الاستعلامات تلقائيًا بالشركة الحالية.

### مثال على النطاق العام

```php
// app/Models/Company.php
protected static function booted()
{
    static::addGlobalScope('company', function (Builder $builder) {
        if (auth()->check() && !auth()->user()->isSuperAdmin()) {
            $builder->where('id', auth()->user()->company_id);
        }
    });
}
```

## نظام الصلاحيات

### المستويات

النظام يدعم ثلاثة مستويات من الصلاحيات:

1. **مستوى النظام**: صلاحيات على مستوى النظام بأكمله (للمشرفين)
2. **مستوى الشركة**: صلاحيات على مستوى شركة معينة
3. **مستوى الفرع**: صلاحيات على مستوى فرع معين

### تنفيذ الصلاحيات

يتم تنفيذ الصلاحيات من خلال:

1. **الأدوار (Roles)**: مجموعات من الصلاحيات يمكن تعيينها للمستخدمين
2. **الصلاحيات (Permissions)**: إذن محدد للقيام بعملية معينة
3. **علاقة العديد للعديد**: يمكن للمستخدم أن يكون له عدة أدوار، وكل دور يمكن أن يحتوي على عدة صلاحيات

### التحقق من الصلاحيات

```php
// التحقق من صلاحية معينة
if (auth()->user()->hasPermission('create-elevator')) {
    // القيام بالعملية
}

// التحقق من دور معين
if (auth()->user()->hasRole('company-admin')) {
    // القيام بالعملية
}
```

## العلاقات بين النماذج

### العلاقات الرئيسية

1. **الشركة والفروع**:
```php
// app/Models/Company.php
public function branches()
{
    return $this->hasMany(Branch::class);
}

// app/Models/Branch.php
public function company()
{
    return $this->belongsTo(Company::class);
}
```

2. **المستخدم والشركة والفرع**:
```php
// app/Models/User.php
public function company()
{
    return $this->belongsTo(Company::class);
}

public function branch()
{
    return $this->belongsTo(Branch::class);
}
```

3. **المصعد والعميل والموقع**:
```php
// app/Models/Elevator.php
public function customer()
{
    return $this->belongsTo(Customer::class);
}

public function location()
{
    return $this->belongsTo(Location::class);
}
```

4. **عقد الصيانة والمصاعد**:
```php
// app/Models/MaintenanceContract.php
public function elevators()
{
    return $this->belongsToMany(Elevator::class);
}

// app/Models/Elevator.php
public function maintenanceContracts()
{
    return $this->belongsToMany(MaintenanceContract::class);
}
```

## تدفق العمل النموذجي

### إضافة ميزة جديدة

1. **إنشاء الهجرة**: إنشاء ملف هجرة لأي جداول جديدة مطلوبة
```bash
php artisan make:migration create_new_feature_table
```

2. **إنشاء النموذج**: إنشاء نموذج للكيان الجديد
```bash
php artisan make:model NewFeature
```

3. **إنشاء وحدة التحكم**: إنشاء وحدة تحكم للتعامل مع الطلبات
```bash
php artisan make:controller NewFeatureController --resource
```

4. **تعريف المسارات**: إضافة المسارات في ملف المسارات المناسب
```php
// routes/web.php
Route::resource('new-features', NewFeatureController::class);
```

5. **إنشاء العروض**: إنشاء ملفات العرض في مجلد resources/views

6. **إضافة الصلاحيات**: تحديث بذور الصلاحيات لإضافة الصلاحيات الجديدة
```php
// database/seeders/PermissionsSeeder.php
Permission::create(['name' => 'view-new-feature', 'level' => 'company']);
Permission::create(['name' => 'create-new-feature', 'level' => 'company']);
Permission::create(['name' => 'edit-new-feature', 'level' => 'company']);
Permission::create(['name' => 'delete-new-feature', 'level' => 'company']);
```

## أفضل الممارسات

### 1. استخدام نمط Repository

لفصل منطق الوصول إلى البيانات عن وحدات التحكم:

```php
// app/Repositories/ElevatorRepository.php
class ElevatorRepository
{
    public function getAll($companyId)
    {
        return Elevator::where('company_id', $companyId)->get();
    }
    
    public function findById($id, $companyId)
    {
        return Elevator::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();
    }
    
    public function create(array $data)
    {
        return Elevator::create($data);
    }
    
    public function update($id, array $data, $companyId)
    {
        $elevator = $this->findById($id, $companyId);
        $elevator->update($data);
        return $elevator;
    }
    
    public function delete($id, $companyId)
    {
        $elevator = $this->findById($id, $companyId);
        return $elevator->delete();
    }
    
    public function getByCustomer($customerId, $companyId)
    {
        return Elevator::where('company_id', $companyId)
            ->where('customer_id', $customerId)
            ->get();
    }
    
    public function getByLocation($locationId, $companyId)
    {
        return Elevator::where('company_id', $companyId)
            ->where('location_id', $locationId)
            ->get();
    }
    
    public function getWithMaintenanceContracts($companyId)
    {
        return Elevator::where('company_id', $companyId)
            ->with('maintenanceContracts')
            ->get();
    }
}
```

### 2. التحقق من الصحة

استخدم طلبات النماذج للتحقق من صحة البيانات:

```php
// app/Http/Requests/ElevatorRequest.php
class ElevatorRequest extends FormRequest
{
    public function rules()
    {
        return [
            'model' => 'required|string|max:255',
            'serial_number' => 'required|string|max:255|unique:elevators,serial_number,' . $this->elevator,
            'capacity' => 'required|numeric|min:1',
            'floors' => 'required|integer|min:2',
            'customer_id' => 'required|exists:customers,id',
            'location_id' => 'required|exists:locations,id',
        ];
    }
}
```

### 3. استخدام الوسائط

استخدم الوسائط للتحقق من الصلاحيات:

```php
// app/Http/Middleware/CheckPermission.php
class CheckPermission
{
    public function handle($request, Closure $next, $permission)
    {
        if (!auth()->user()->hasPermission($permission)) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه الصفحة');
        }
        
        return $next($request);
    }
}
```

### 4. كتابة الاختبارات

اكتب اختبارات لكل وظيفة جديدة:

```php
// tests/Feature/ElevatorTest.php
class ElevatorTest extends TestCase
{
    use RefreshDatabase;
    
    public function test_user_can_view_elevators()
    {
        $user = User::factory()->create();
        $this->actingAs($user);
        
        $response = $this->get('/elevators');
        
        $response->assertStatus(200);
    }
    
    public function test_user_can_create_elevator()
    {
        $user = User::factory()->create();
        $this->actingAs($user);
        
        $response = $this->post('/elevators', [
            'model' => 'Test Model',
            'serial_number' => 'SN12345',
            'capacity' => 1000,
            'floors' => 10,
            'customer_id' => 1,
            'location_id' => 1,
        ]);
        
        $response->assertRedirect('/elevators');
        $this->assertDatabaseHas('elevators', ['model' => 'Test Model']);
    }
}
```

## حل المشاكل الشائعة

### 1. مشاكل الصلاحيات

إذا واجهت مشاكل في الصلاحيات:

1. تأكد من تشغيل سكريبت `give_permissions.php`
2. تحقق من تعيين الأدوار للمستخدمين
3. تحقق من تعيين الصلاحيات للأدوار
4. استخدم الأمر التالي لعرض الصلاحيات المتاحة:
```php
dd(auth()->user()->getAllPermissions());
```

### 2. مشاكل عزل البيانات

إذا كان المستخدم يرى بيانات شركات أخرى:

1. تأكد من تطبيق النطاق العام على النموذج
2. تحقق من تعيين `company_id` بشكل صحيح
3. تأكد من تفعيل وسيط `CheckCompany`

### 3. مشاكل الهجرات

إذا فشلت الهجرات بسبب قيود المفاتيح الخارجية:

```php
// في ملف الهجرة
public function up()
{
    Schema::disableForeignKeyConstraints();
    
    // إنشاء الجداول
    
    Schema::enableForeignKeyConstraints();
}
```

## الخطوات التالية للتطوير

### 1. استكمال نظام إدارة المخزون المتقدم

- تطوير نظام الباركود/QR Code لتتبع المخزون
- تنفيذ نظام طلبات الشراء التلقائية
- تحسين نظام تقييم المخزون
- إضافة تقارير تحليلية للمخزون

### 2. تطوير نظام إدارة المشاريع والتركيبات

- إنشاء نماذج البيانات الأساسية للمشاريع
- تطوير واجهات المستخدم لإدارة المشاريع
- تنفيذ جدولة مراحل المشاريع
- إضافة نظام تتبع التقدم في المشاريع
- تطوير نظام إدارة الموارد للمشاريع

### 3. تطوير نظام إدارة الموارد البشرية

- إنشاء نماذج البيانات الأساسية للموظفين
- تطوير نظام الحضور والانصراف
- تنفيذ نظام إدارة الإجازات
- تطوير نظام الرواتب والحوافز
- إضافة نظام تقييم الأداء

### 4. تطوير تطبيق الجوال

- تطوير تطبيق للفنيين لتسجيل أعمال الصيانة
- تطوير تطبيق للعملاء لمتابعة المصاعد والعقود
- تنفيذ نظام الإشعارات للجوال
- ربط التطبيق بنظام تحديد المواقع GPS

## تطوير قوالب الفواتير

### هيكل نظام قوالب الفواتير

نظام قوالب الفواتير يتكون من المكونات التالية:

1. **نموذج البيانات**: `app/Models/InvoiceTemplate.php`
2. **وحدة التحكم**: `app/Http/Controllers/InvoiceTemplateController.php`
3. **ملفات العرض**:
   - `resources/views/invoice_templates/index.blade.php`
   - `resources/views/invoice_templates/create.blade.php`
   - `resources/views/invoice_templates/edit.blade.php`
   - `resources/views/invoice_templates/show.blade.php`
4. **المسارات**: تعريفات المسارات في `routes/invoice.php`

### نموذج البيانات (InvoiceTemplate)

نموذج `InvoiceTemplate` يحتوي على الحقول التالية:

```php
/**
 * حقول نموذج قالب الفاتورة
 *
 * @property int $id
 * @property int $company_id
 * @property string $name
 * @property string $language
 * @property string|null $header_html
 * @property string|null $footer_html
 * @property string|null $css_styles
 * @property bool $is_default
 * @property bool $is_active
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 */
```

### إضافة حقول جديدة لقوالب الفواتير

لإضافة حقول جديدة لقوالب الفواتير، اتبع الخطوات التالية:

1. **إنشاء ملف هجرة جديد**:
   ```bash
   php artisan make:migration add_new_field_to_invoice_templates_table
   ```

2. **تعديل ملف الهجرة**:
   ```php
   public function up()
   {
       Schema::table('invoice_templates', function (Blueprint $table) {
           $table->string('new_field')->nullable()->after('css_styles');
       });
   }

   public function down()
   {
       Schema::table('invoice_templates', function (Blueprint $table) {
           $table->dropColumn('new_field');
       });
   }
   ```

3. **تنفيذ الهجرة**:
   ```bash
   php artisan migrate
   ```

4. **تحديث نموذج البيانات**:
   ```php
   protected $fillable = [
       'company_id',
       'name',
       'language',
       'header_html',
       'footer_html',
       'css_styles',
       'new_field',
       'is_default',
       'is_active',
   ];
   ```

5. **تحديث وحدة التحكم والنماذج**:
   - إضافة التحقق من صحة الحقل الجديد في طرق `store` و `update`
   - تحديث نماذج الإنشاء والتعديل لإضافة الحقل الجديد

### تخصيص عملية توليد HTML للفواتير

يتم توليد HTML للفواتير في طريقة `generatePreviewHtml` في وحدة التحكم `InvoiceTemplateController`. لتخصيص هذه العملية:

1. **تعديل طريقة `generatePreviewHtml`**:
   ```php
   protected function generatePreviewHtml($template, $data)
   {
       // تخصيص عملية توليد HTML
       // ...
   }
   ```

2. **إضافة متغيرات جديدة للقالب**:
   ```php
   $data['custom_variable'] = 'قيمة مخصصة';
   ```

3. **استخدام المتغيرات في القالب**:
   ```html
   <div>{{ $custom_variable }}</div>
   ```

### إضافة ميزات جديدة لقوالب الفواتير

#### 1. إضافة دعم للمتغيرات الديناميكية

يمكن إضافة دعم للمتغيرات الديناميكية في قوالب الفواتير باستخدام نظام استبدال النصوص:

```php
// تعريف المتغيرات المتاحة
$variables = [
    '{{company_name}}' => $data['company']['name'],
    '{{invoice_number}}' => $data['invoice']['invoice_number'],
    // ...
];

// استبدال المتغيرات في HTML
$html = $template->header_html;
foreach ($variables as $key => $value) {
    $html = str_replace($key, $value, $html);
}
```

#### 2. إضافة دعم للأقسام الشرطية

يمكن إضافة دعم للأقسام الشرطية في قوالب الفواتير:

```php
// تعريف نمط للأقسام الشرطية
$pattern = '/\{\{if\s+(.+?)\}\}(.*?)\{\{endif\}\}/s';

// معالجة الأقسام الشرطية
preg_replace_callback($pattern, function($matches) use ($data) {
    $condition = $matches[1];
    $content = $matches[2];
    
    // تقييم الشرط
    $result = eval("return " . $this->parseCondition($condition, $data) . ";");
    
    return $result ? $content : '';
}, $html);
```

#### 3. إضافة دعم للحلقات التكرارية

يمكن إضافة دعم للحلقات التكرارية في قوالب الفواتير:

```php
// تعريف نمط للحلقات التكرارية
$pattern = '/\{\{foreach\s+(.+?)\}\}(.*?)\{\{endforeach\}\}/s';

// معالجة الحلقات التكرارية
preg_replace_callback($pattern, function($matches) use ($data) {
    $array = $matches[1];
    $content = $matches[2];
    $result = '';
    
    // الحصول على المصفوفة من البيانات
    $items = $this->getArrayFromData($array, $data);
    
    // تكرار المحتوى لكل عنصر
    foreach ($items as $item) {
        $itemContent = $content;
        // استبدال متغيرات العنصر
        // ...
        $result .= $itemContent;
    }
    
    return $result;
}, $html);
```

### أفضل الممارسات لتطوير قوالب الفواتير

1. **استخدام CSS المضمن**: استخدم CSS المضمن في القالب لضمان عرض الفاتورة بشكل صحيح عند الطباعة أو التصدير إلى PDF.

2. **اختبار التوافق مع المتصفحات**: تأكد من اختبار القوالب في مختلف المتصفحات للتأكد من توافقها.

3. **دعم الاتجاهات RTL و LTR**: تأكد من أن القوالب تدعم كلا الاتجاهين للغات العربية والإنجليزية.

4. **استخدام وحدات القياس النسبية**: استخدم وحدات قياس نسبية (مثل % أو em) بدلاً من وحدات مطلقة (مثل px) لضمان التوافق مع مختلف أحجام الصفحات.

5. **فصل المنطق عن العرض**: حاول فصل منطق توليد البيانات عن قوالب العرض قدر الإمكان.

### 3. تطوير نظام إدارة المخزون

- إنشاء نظام إدارة المستودعات المتعددة
- تطوير تتبع الدفعات والأرقام التسلسلية
- إضافة نظام الباركود/QR Code
- تطوير طلبات الشراء التلقائية

### 4. تطوير نظام إدارة علاقات العملاء (CRM)

- تطوير لوحة متابعة الفرص التجارية
- إضافة نظام التذكير بمتابعة العملاء
- تطوير تقارير أداء المبيعات
- إضافة نظام تسجيل المكالمات والاجتماعات

### 5. تطوير نظام الولاء والمكافآت

- تطوير آلية احتساب النقاط التلقائية
- إضافة بوابة العملاء لاستعراض النقاط والمكافآت
- تطوير نظام الإشعارات للعملاء عن المكافآت الجديدة
- إضافة تقارير تحليلية لفعالية برنامج الولاء

## موارد إضافية

### وثائق Laravel
- [الموقع الرسمي لـ Laravel](https://laravel.com/docs)
- [Laracasts](https://laracasts.com/)

### أدوات مفيدة
- [Laravel Debugbar](https://github.com/barryvdh/laravel-debugbar)
- [Laravel IDE Helper](https://github.com/barryvdh/laravel-ide-helper)
- [Laravel Telescope](https://laravel.com/docs/telescope)

### مجتمع المطورين
- [Laravel News](https://laravel-news.com/)
- [Laravel.io](https://laravel.io/forum)
- [Laracasts Forum](https://laracasts.com/discuss)